
package com.smartgwt.client.docs;

/**
 * <h3>GraalJS Polyglot API for DMI</h3>
 * <h3>Using GraalJS Polyglot API from DMI</h3>
 *  <P>
 * The {@link com.smartgwt.client.docs.ServerScript Server Scripting} documentation describes how
 * to use JavaScript
 *  (via GraalJS or Nashorn) through the standard JSR223 ScriptEngine API.  This approach is
 *  ideal for inline scripts in .ds.xml files and Velocity templates.
 *  <P>
 *  For more advanced use cases - such as loading complex JavaScript libraries, maintaining
 *  stateful JavaScript contexts across requests, or integrating JavaScript-based processing
 *  into your DMI logic - you can use GraalJS's Polyglot API directly.  This provides finer
 *  control over the JavaScript execution environment.
 *  <P>
 *  <h4>Setup: Adding GraalJS to Your Project</h4>
 *  <P>
 *  To use GraalJS on a standard Oracle or OpenJDK JVM (not full GraalVM), add these Maven
 *  dependencies:
 *  <pre>
 *      &lt;dependency&gt;
 *          &lt;groupId&gt;org.graalvm.polyglot&lt;/groupId&gt;
 *          &lt;artifactId&gt;polyglot&lt;/artifactId&gt;
 *          &lt;version&gt;24.1.1&lt;/version&gt;
 *      &lt;/dependency&gt;
 *      &lt;dependency&gt;
 *          &lt;groupId&gt;org.graalvm.polyglot&lt;/groupId&gt;
 *          &lt;artifactId&gt;js&lt;/artifactId&gt;
 *          &lt;version&gt;24.1.1&lt;/version&gt;
 *          &lt;type&gt;pom&lt;/type&gt;
 *      &lt;/dependency&gt;
 *  </pre>
 * Or download the standalone JARs from <a href='https://www.graalvm.org/downloads/'
 * target='_blank'>https://www.graalvm.org/downloads/</a>.
 *  Smart GWT also ships GraalJS libraries in <code>WEB-INF/lib-graaljs/</code> which can be
 *  unzipped and added to your classpath.
 *  <P>
 *  <h4>Basic DMI Example: Calling JavaScript from Java</h4>
 *  <P>
 *  Here is a simple DMI class that uses GraalJS to execute JavaScript code:
 *  <pre>
 *  package com.mycompany.dmi;
 * 
 *  import org.graalvm.polyglot.*;
 *  import com.isomorphic.datasource.*;
 *  import java.util.Map;
 * 
 *  public class JavaScriptProcessor {
 * 
 *      // Reusable context for better performance (create once, reuse)
 *      private static final Engine engine = Engine.create();
 * 
 *      &#47;**
 *       * DMI method that processes a record using JavaScript logic.
 *       * Called from a DataSource operationBinding.
 *       *&#47;
 *      public DSResponse processWithJS(DSRequest dsRequest) throws Exception {
 * 
 *          // Create a context with host access enabled (allows JS to call Java)
 *          try (Context context = Context.newBuilder("js")
 *                  .engine(engine)
 *                  .allowHostAccess(HostAccess.ALL)
 *                  .allowHostClassLookup(className -> true)
 *                  .build()) {
 * 
 *              // Get the record data from the request
 *              Map record = dsRequest.getValues();
 * 
 *              // Make the record available to JavaScript as 'record'
 *              context.getBindings("js").putMember("record", record);
 * 
 *              // Execute JavaScript that processes the record
 *              Value result = context.eval("js",
 *                  "// JavaScript code here\n" +
 *                  "var processed = {};\n" +
 *                  "processed.name = record.get('firstName') + ' ' + record.get('lastName');\n" +
 *                  "processed.nameLength = processed.name.length;\n" +
 *                  "processed;\n"  // Return the result
 *              );
 * 
 *              // Convert JavaScript result back to Java Map
 *              Map processedRecord = result.as(Map.class);
 * 
 *              // Return the processed data
 *              DSResponse response = new DSResponse();
 *              response.setData(processedRecord);
 *              return response;
 *          }
 *      }
 *  }
 *  </pre>
 *  <P>
 *  <h4>DataSource Configuration</h4>
 *  <P>
 *  Configure your DataSource to call this DMI:
 *  <pre>
 *      &lt;DataSource ID="myDataSource"&gt;
 *          &lt;fields&gt;
 *              &lt;field name="firstName" type="text"/&gt;
 *              &lt;field name="lastName" type="text"/&gt;
 *          &lt;/fields&gt;
 *          &lt;operationBindings&gt;
 *              &lt;operationBinding operationType="custom" operationId="processJS"&gt;
 *                  &lt;serverObject className="com.mycompany.dmi.JavaScriptProcessor"/&gt;
 *                  &lt;methodArguments&gt;$dsRequest&lt;/methodArguments&gt;
 *              &lt;/operationBinding&gt;
 *          &lt;/operationBindings&gt;
 *      &lt;/DataSource&gt;
 *  </pre>
 *  <P>
 *  <h4>Loading External JavaScript Files</h4>
 *  <P>
 *  For more complex logic, load JavaScript from files:
 *  <pre>
 *  public class JavaScriptProcessor {
 * 
 *      private static final Engine engine = Engine.create();
 * 
 *      public DSResponse processWithExternalJS(DSRequest dsRequest) throws Exception {
 * 
 *          try (Context context = Context.newBuilder("js")
 *                  .engine(engine)
 *                  .allowHostAccess(HostAccess.ALL)
 *                  .allowHostClassLookup(className -> true)
 *                  .allowIO(IOAccess.ALL)  // Required to load files
 *                  .build()) {
 * 
 *              // Load a JavaScript file from the classpath or filesystem
 *              String scriptPath = "/path/to/my-script.js";
 *              Source source = Source.newBuilder("js", new File(scriptPath)).build();
 *              context.eval(source);
 * 
 *              // Call a function defined in the loaded script
 *              Value processFunc = context.getBindings("js").getMember("processRecord");
 *              Value result = processFunc.execute(dsRequest.getValues());
 * 
 *              DSResponse response = new DSResponse();
 *              response.setData(result.as(Map.class));
 *              return response;
 *          }
 *      }
 *  }
 *  </pre>
 *  <P>
 *  <h4>Performance Considerations</h4>
 *  <P>
 *  <ul>
 *  <li><b>Reuse the Engine:</b> Creating an <code>Engine</code> is expensive.  Create it once
 *  (as a static field) and reuse it across contexts.</li>
 *  <li><b>Context Pooling:</b> For high-throughput scenarios, consider pooling
 *  <code>Context</code> objects rather than creating new ones per request.</li>
 *  <li><b>Warm-up:</b> GraalJS compiles JavaScript to optimized machine code over time.
 *  Initial calls may be slower than subsequent ones.</li>
 *  </ul>
 *  <P>
 *  <h4>Security Notes</h4>
 *  <P>
 *  The example above uses <code>allowHostAccess(HostAccess.ALL)</code> which gives JavaScript
 *  full access to Java classes.  For production use with untrusted scripts, consider:
 *  <ul>
 *  <li>Using <code>HostAccess.EXPLICIT</code> and annotating allowed methods with
 *  <code>@HostAccess.Export</code></li>
 *  <li>Restricting <code>allowHostClassLookup</code> to specific packages</li>
 *  <li>Disabling <code>allowIO</code> if file access is not needed</li>
 *  </ul>
 *  <P>
 *  For the JSR223-based approach that integrates with Smart GWT's built-in scripting
 *  infrastructure, see {@link com.smartgwt.client.docs.ServerScript Server Scripting}.
 */
public interface GraalPolyglotDMI {
}
