package com.smartgwt.sample.showcase.client.grid;

import com.google.gwt.user.client.Timer;
import com.smartgwt.client.data.DSRequest;
import com.smartgwt.client.data.DSResponse;
import com.smartgwt.client.data.DataSource;
import com.smartgwt.client.data.Hilite;
import com.smartgwt.client.data.Criterion;
import com.smartgwt.client.data.Record;
import com.smartgwt.client.data.ResponseTransformer;
import com.smartgwt.client.widgets.IButton;
import com.smartgwt.client.types.ListGridFieldType;
import com.smartgwt.client.types.OperatorId;
import com.smartgwt.client.types.DSOperationType;
import com.smartgwt.client.widgets.Canvas;
import com.smartgwt.client.widgets.Label;
import com.smartgwt.client.widgets.grid.ListGrid;
import com.smartgwt.client.widgets.grid.ListGridField;
import com.smartgwt.client.widgets.events.ClickEvent;
import com.smartgwt.client.widgets.events.ClickHandler;
import com.smartgwt.client.widgets.layout.VLayout;
import com.smartgwt.sample.showcase.client.PanelFactory;
import com.smartgwt.sample.showcase.client.ShowcasePanel;
import com.smartgwt.sample.showcase.client.SourceEntity;

public class FetchingVisibleOnlySample extends ShowcasePanel {

    private static final String DESCRIPTION = "This example uses <code>listGrid.fetchFields</code> "+
        "to limit the set of fields retrieved from "+
        "its dataSource. Limiting the set of fields allows an application to communicate efficiently "+
        "with large data tables."+
        "<P>"+
        "When the <code>fetchFields</code> feature is active the grid will always fetch values for all "+
        "visible fields. You can also configure it to fetch additional fields, making them available "+
        "for custom application logic. In this case the \"continent\" field is fetched along with the "+
        "visible fields, so the grid can use it to govern which records may be edited."+
        "<P>"+
        "As you interact with the grid, if built-in features require field values that have not been "+
        "retrieved, they will be fetched automatically."+
        "<P>"+
        "To see this in action, right click and display the hidden \"Population\" column. The grid will "+
        "re-fetch its data to pick up the newly visible field values."+
        "<P>"+
        "Similarly if you apply a highlight to the grid that requires values for a hidden field, "+
        "they will be fetched automatically. Click the button to highlight G8 member countries. "+
        "When the highlight is applied, the grid recognizes that it needs the extra field and fetches it.";

    public static class Factory implements PanelFactory {
        private String id;

        public ShowcasePanel create() {
            FetchingVisibleOnlySample panel = new FetchingVisibleOnlySample();
            id = panel.getID();
            return panel;
        }

        public String getID() {
            return id;
        }

        public String getDescription() {
            return DESCRIPTION;
        }
    }

    private Label outputFields;
    private ListGrid countryGrid;
    private VLayout vLayout;
    
    private class ServerCountLabel extends Label {
        private int count = 0;  
  
        public ServerCountLabel() {  
            setWidth(800);
            setHeight(40);
            setPadding(10);
            setBorder("1px solid grey");
            setContents("<b>List of requested fields: NA</b>");
        }  
  
        public void updateContent (String outputs, int totalRows, int startRow, int endRow) {
            count++;
            outputFields.setContents("Number of server trips: " + this.count + 
                                     "<br/>Total rows in this filter set: " + totalRows + 
                                     "<br/>Last range of records returned: " + startRow + " to " + endRow + 
                                     "<br/>Fields included in returned records: <b>"+outputs+"</b>");
        }
    }
    
    public Canvas getViewPanel() {

        vLayout = new VLayout();
        vLayout.setWidth(800);
        vLayout.setHeight(250);
        vLayout.setMembersMargin(10);
    	
        IButton highlightButton = new IButton();
        highlightButton.setAutoFit(true);
        highlightButton.setTitle("Highlight G8 Countries");
        highlightButton.addClickHandler(new ClickHandler() {
            @Override
            public void onClick(ClickEvent event) {
                countryGrid.destroy();
                recreateListGrid(true);
            }
        });
   
        outputFields = new ServerCountLabel();
    	    	
        vLayout.setMembers(highlightButton, outputFields);
    	
        recreateListGrid(false);

        return vLayout;
    }

    private void recreateListGrid (boolean includeHilites) {
        Hilite hilites[] = new Hilite[] {  
            new Hilite() {{  
                setFieldNames("countryName");  
                setCriteria(new Criterion("member_g8", OperatorId.EQUALS, true));  
                setCssText("background-color:#FFA07A;");  
            }}  
        };
        
        DataSource countryDS = DataSource.get("countryDS", null, new ResponseTransformer() {
            protected void transformResponse(DSResponse response, DSRequest request, Object data) {
                if(request.getOperationType() == DSOperationType.FETCH) {
                    int totalRows = response.getTotalRows(); 
                    int startRow = response.getStartRow(); 
                    int endRow = response.getEndRow(); 
                    String outouts = request.getOutputs();
                    ((ServerCountLabel) outputFields).updateContent(outouts, totalRows, startRow, endRow); 
                    outputFields.setBackgroundColor("#ffff77"); 
                    new Timer() { 
                        public void run() { 
                            outputFields.setBackgroundColor("#ffffff"); 
                        } 
                    }.schedule(500); 
                }
                defaultTransformResponse(response, request, data);
            }
        });
        
        countryGrid = new ListGrid() {
            public Boolean canEditCell(int rowNum, int colNum) {
                Record record = countryGrid.getRecord(rowNum);
                String fieldName = countryGrid.getFieldName(colNum);
                if (fieldName.equals("countryCode") || (fieldName.equals("capital") && record.getAttribute("continent").equals("Europe"))) {
                    return false;
                }
                return true;
            };
    	    
        };
    	
        countryGrid.setWidth100();
        countryGrid.setHeight(300);
        countryGrid.setShowFilterEditor(true);
        countryGrid.setCanEditHilites(true);
        countryGrid.setAutoFetchData(true);
        countryGrid.setDataSource(countryDS);
    	
        ListGridField countryCode = new ListGridField("countryCode", "Flag");
        countryCode.setWidth(40);
        countryCode.setType(ListGridFieldType.IMAGE);
        countryCode.setImageURLPrefix("flags/16/");
        countryCode.setImageURLSuffix(".png");
        countryCode.setCanEdit(false);
        ListGridField countryName = new ListGridField("countryName");
        ListGridField capital = new ListGridField("capital");
        ListGridField population = new ListGridField("population", "Population");
        population.setHidden(true);
        ListGridField continent = new ListGridField("continent");
    	
        countryGrid.setDefaultFields(countryCode,countryName,capital,population);
        countryGrid.setFetchFields(continent);
    	
        if (includeHilites) countryGrid.setHilites(hilites);

        vLayout.addMember(countryGrid, 1);
    }
    
    public String getIntro() {
        return DESCRIPTION;
    }
    
}