/*

  SmartClient Ajax RIA system
  Version v14.0p_2025-11-05/LGPL Deployment (2025-11-05)

  Copyright 2000 and beyond Isomorphic Software, Inc. All rights reserved.
  "SmartClient" is a trademark of Isomorphic Software, Inc.

  LICENSE NOTICE
     INSTALLATION OR USE OF THIS SOFTWARE INDICATES YOUR ACCEPTANCE OF
     ISOMORPHIC SOFTWARE LICENSE TERMS. If you have received this file
     without an accompanying Isomorphic Software license file, please
     contact licensing@isomorphic.com for details. Unauthorized copying and
     use of this software is a violation of international copyright law.

  DEVELOPMENT ONLY - DO NOT DEPLOY
     This software is provided for evaluation, training, and development
     purposes only. It may include supplementary components that are not
     licensed for deployment. The separate DEPLOY package for this release
     contains SmartClient components that are licensed for deployment.

  PROPRIETARY & PROTECTED MATERIAL
     This software contains proprietary materials that are protected by
     contract and intellectual property law. You are expressly prohibited
     from attempting to reverse engineer this software or modify this
     software for human readability.

  CONTACT ISOMORPHIC
     For more information regarding license rights and restrictions, or to
     report possible license violations, please contact Isomorphic Software
     by email (licensing@isomorphic.com) or web (www.isomorphic.com).

*/
//> @object Promise
// Extensions to the <code>Promise</code> class, or a polyfill of the ES6 <code>Promise</code>
// class if not provided natively by the browser.
// @treeLocation Client Reference/System
// @visibility external
//<


// BEGIN taylorhakes/promise-polyfill
// The following code through "END taylorhakes/promise-polyfill" is a modified version of:
// https://raw.githubusercontent.com/taylorhakes/promise-polyfill/c9e76c1f60654e5f35b399094ec616f54720134d/dist/polyfill.js
// .. which is distributed under the following license:




isc.Promise = (function () {
//!DONTOBFUSCATE
'use strict';

/**
 * @this {Promise}
 */
function finallyConstructor(callback) {
  var constructor = this.constructor;
  return this.then(
    function(value) {
      // @ts-ignore
      return constructor.resolve(callback()).then(function() {
        return value;
      });
    },
    function(reason) {
      // @ts-ignore
      return constructor.resolve(callback()).then(function() {
        // @ts-ignore
        return constructor.reject(reason);
      });
    }
  );
}

function allSettled(arr) {
  var P = this;
  return new P(function(resolve, reject) {
    if (!(arr && typeof arr.length !== 'undefined')) {
      return reject(
        new TypeError(
          typeof arr +
            ' ' +
            arr +
            ' is not iterable(cannot read property Symbol(Symbol.iterator))'
        )
      );
    }
    var args = Array.prototype.slice.call(arr);
    if (args.length === 0) return resolve([]);
    var remaining = args.length;

    function res(i, val) {
      if (val && (typeof val === 'object' || typeof val === 'function')) {
        var then = val.then;
        if (typeof then === 'function') {
          then.call(
            val,
            function(val) {
              res(i, val);
            },
            function(e) {
              args[i] = { status: 'rejected', reason: e };
              if (--remaining === 0) {
                resolve(args);
              }
            }
          );
          return;
        }
      }
      args[i] = { status: 'fulfilled', value: val };
      if (--remaining === 0) {
        resolve(args);
      }
    }

    for (var i = 0; i < args.length; i++) {
      res(i, args[i]);
    }
  });
}

/**
 * @constructor
 */
function AggregateError(errors, message) {
  this.name = 'AggregateError', this.errors = errors;
  this.message = message || '';
}
AggregateError.prototype = Error.prototype;

function any(arr) {
  var P = this;
  return new P(function(resolve, reject) {
    if (!(arr && typeof arr.length !== 'undefined')) {
      return reject(new TypeError('Promise.any accepts an array'));
    }

    var args = Array.prototype.slice.call(arr);
    if (args.length === 0) return reject();

    var rejectionReasons = [];
    for (var i = 0; i < args.length; i++) {
      try {
        P.resolve(args[i])
          .then(resolve)['catch'](function(error) {
            rejectionReasons.push(error);
            if (rejectionReasons.length === args.length) {
              reject(
                new AggregateError(
                  rejectionReasons,
                  'All promises were rejected'
                )
              );
            }
          });
      } catch (ex) {
        reject(ex);
      }
    }
  });
}

/**
 * @constructor
 * @param {Function} fn
 */
function Promise(fn) {
  if (!(this instanceof Promise))
    throw new TypeError('Promises must be constructed via new');
  if (typeof fn !== 'function') throw new TypeError('not a function');
  /** @type {!number} */
  this._state = 0;
  /** @type {!boolean} */
  this._handled = false;
  /** @type {Promise|undefined} */
  //this._value = undefined;
  /** @type {!Array<!Function>} */
  this._deferreds = [];

  doResolve(fn, this);
}

function handle(self, deferred) {
  while (self._state === 3) {
    self = self._value;
  }
  if (self._state === 0) {
    self._deferreds.push(deferred);
    return;
  }
  self._handled = true;
  Promise._immediateFn(function() {
    var cb = self._state === 1 ? deferred.onFulfilled : deferred.onRejected;
    if (cb === null) {
      (self._state === 1 ? resolve : reject)(deferred.promise, self._value);
      return;
    }
    var ret;
    try {
      ret = cb(self._value);
    } catch (e) {
      reject(deferred.promise, e);
      return;
    }
    resolve(deferred.promise, ret);
  });
}

function resolve(self, newValue) {
  try {
    // Promise Resolution Procedure: https://github.com/promises-aplus/promises-spec#the-promise-resolution-procedure
    if (newValue === self)
      throw new TypeError('A promise cannot be resolved with itself.');
    if (
      newValue &&
      (typeof newValue === 'object' || typeof newValue === 'function')
    ) {
      var then = newValue.then;
      if (newValue instanceof Promise) {
        self._state = 3;
        self._value = newValue;
        finale(self);
        return;
      } else if (typeof then === 'function') {
        doResolve(then.bind(newValue), self);
        return;
      }
    }
    self._state = 1;
    self._value = newValue;
    finale(self);
  } catch (e) {
    reject(self, e);
  }
}

function reject(self, newValue) {
  self._state = 2;
  self._value = newValue;
  finale(self);
}

function finale(self) {
  if (self._state === 2 && self._deferreds.length === 0) {
    Promise._immediateFn(function() {
      if (!self._handled) {
        Promise._unhandledRejectionFn(self._value);
      }
    });
  }

  for (var i = 0, len = self._deferreds.length; i < len; i++) {
    handle(self, self._deferreds[i]);
  }
  self._deferreds = null;
}

/**
 * @constructor
 */
function Handler(onFulfilled, onRejected, promise) {
  this.onFulfilled = typeof onFulfilled === 'function' ? onFulfilled : null;
  this.onRejected = typeof onRejected === 'function' ? onRejected : null;
  this.promise = promise;
}

/**
 * Take a potentially misbehaving resolver function and make sure
 * onFulfilled and onRejected are only called once.
 *
 * Makes no guarantees about asynchrony.
 */
function doResolve(fn, self) {
  var done = false;
  try {
    fn(
      function(value) {
        if (done) return;
        done = true;
        resolve(self, value);
      },
      function(reason) {
        if (done) return;
        done = true;
        reject(self, reason);
      }
    );
  } catch (ex) {
    if (done) return;
    done = true;
    reject(self, ex);
  }
}

Promise.prototype['catch'] = function(onRejected) {
  return this.then(null, onRejected);
};

Promise.prototype.then = function(onFulfilled, onRejected) {
  // @ts-ignore
  var prom = new this.constructor(isc.Class.NO_OP);

  handle(this, new Handler(onFulfilled, onRejected, prom));
  return prom;
};

Promise.prototype['finally'] = finallyConstructor;

Promise.all = function(arr) {
  return new Promise(function(resolve, reject) {
    if (!isc.isAn.Array(arr)) {
      return reject(new TypeError('Promise.all accepts an array'));
    }

    var args = Array.prototype.slice.call(arr);
    if (args.length === 0) return resolve([]);
    var remaining = args.length;

    function res(i, val) {
      try {
        if (val && (typeof val === 'object' || typeof val === 'function')) {
          var then = val.then;
          if (typeof then === 'function') {
            then.call(
              val,
              function(val) {
                res(i, val);
              },
              reject
            );
            return;
          }
        }
        args[i] = val;
        if (--remaining === 0) {
          resolve(args);
        }
      } catch (ex) {
        reject(ex);
      }
    }

    for (var i = 0; i < args.length; i++) {
      res(i, args[i]);
    }
  });
};

Promise.any = any;

Promise.allSettled = allSettled;

Promise.resolve = function(value) {
  if (value && typeof value === 'object' && value.constructor === Promise) {
    return value;
  }

  return new Promise(function(resolve) {
    resolve(value);
  });
};

Promise.reject = function(value) {
  return new Promise(function(resolve, reject) {
    reject(value);
  });
};

Promise.race = function(arr) {
  return new Promise(function(resolve, reject) {
    if (!isc.isAn.Array(arr)) {
      return reject(new TypeError('Promise.race accepts an array'));
    }

    for (var i = 0, len = arr.length; i < len; i++) {
      Promise.resolve(arr[i]).then(resolve, reject);
    }
  });
};

// Use polyfill for setImmediate for performance gains
Promise._immediateFn =
  // @ts-ignore
  (typeof setImmediate === 'function' &&
    function(fn) {
      // @ts-ignore
      setImmediate(fn);
    }) ||
  function(fn) {
    setTimeout(fn, 0);
  };

Promise._unhandledRejectionFn = function _unhandledRejectionFn(err) {
  isc.logWarn('Possible unhandled Promise rejection: ' + err);
};

var globalNS = window;

// Expose the polyfill if Promise is undefined or set to a
// non-function value. The latter can be due to a named HTMLElement
// being exposed by browsers for legacy reasons.
// https://github.com/taylorhakes/promise-polyfill/issues/114
if (typeof globalNS['Promise'] !== 'function') {
  globalNS['Promise'] = Promise;
} else {
  if (!globalNS.Promise.prototype['finally']) {
    globalNS.Promise.prototype['finally'] = finallyConstructor;
  }
  if (!globalNS.Promise.allSettled) {
    globalNS.Promise.allSettled = allSettled;
  }
  if (!globalNS.Promise.any) {
    globalNS.Promise.any = any;
  }
}

if (typeof globalNS['AggregateError'] !== 'function') {
  globalNS['AggregateError'] = AggregateError;
}

return Promise;
})();


// END taylorhakes/promise-polyfill


Promise.prototype._whenSettled = function (onSettled) {
    return this.then(onSettled, function (rejectReason) {
        return onSettled(isc.defaultAsyncOperationCatchCallback(rejectReason));
    });
};

// Synonym of Promise.prototype.catch() that is more convenient to use.
Promise.prototype._catch = Promise.prototype["catch"];

if (!Promise.prototype["finally"]) {
    // Polyfill of Promise.prototype.finally() for IE<11, which does not support this method:
    // https://caniuse.com/promise-finally
    // Use `["finally"]` because `finally` is a reserved keyword in IE<9.
    Promise.prototype["finally"] = function (onFinally) {
        // https://tc39.es/ecma262/multipage/control-abstraction-objects.html#sec-promise.prototype.finally
        var thenFinally,
            catchFinally;
        if (!isc.isA.Function(onFinally)) {
            catchFinally = thenFinally = onFinally;
        } else {
            thenFinally = function (value) {
                return Promise.resolve(onFinally()).then(function () { return value; });
            };
            catchFinally = function (rejectReason) {
                return Promise.resolve(onFinally()).then(function () { throw rejectReason; });
            };
        }
        return this.then(thenFinally, catchFinally);
    };
}

// Synonym of Promise.prototype.finally() that is more convenient to use.
//
// Because `finally` is a reserved keyword in IE<9, invoking finally() directly causes a syntax
// error in those versions of IE. Also, the work-around syntax ["finally"]() is not very convenient
// because it must be placed on the same line as the code creating/accessing the Promise;
// i.e. this doesn't work as expected:
//
//     this.asyncDoSomething()
//         ["finally"](function () { ... });
//
// (causes `["finally"] is not a function`) because an implicit semicolon is added after the
// call to asyncDoSomething(). Instead, one must use this:
//
//     this.asyncDoSomething()["finally"](function () { ... });
//
// Using _finally(), however, we can do this:
//
//     this.asyncDoSomething()
//         ._finally(function () { ... });
Promise.prototype._finally = Promise.prototype["finally"];

//> @staticMethod Promise.withResolvers()
// Creates a new <code>Promise</code> as well as resolution and rejection functions to resolve
// or reject (respectively) this <code>Promise</code>.
// <p>
// withResolvers() is a polyfill, if not provided natively by the browser, of the
// +externalLink{https://github.com/tc39/proposal-promise-with-resolvers,Promise.withResolvers()}
// TC39 proposal.
// @return (Object) an object having three properties:
// <ul>
// <li><code>resolve</code> - a JavaScript function that takes the value to resolve the linked
// <code>Promise</code> with.
// <li><code>reject</code> - a JavaScript function that takes the reason to reject the
// linked <code>Promise</code>.
// <li><code>promise</code> - the <code>Promise</code> that is fulfilled by the first call to
// either <code>resolve</code> or <code>reject</code>, causing the <code>Promise</code> to
// resolve or reject with the provided value or reason, respectively.
// </ul>
// @visibility external
//<
// https://github.com/tc39/proposal-promise-with-resolvers
if (!Promise.withResolvers) {
    Promise.withResolvers = function () {
        var obj = {},
            P = this;
        obj.promise = new P(function (resolve, reject) {
            obj.resolve = resolve;
            obj.reject = reject;
        });
        return obj;
    };
}

// A utility similar to Promise.allSettled() that additionally takes a cancellationController.
// @param promises (Array of Promise) An array of Promises for AsyncOperationResults.
// @param [cancellationController] (CancellationController) The cancellationController.
// @return (Promise) a <code>Promise</code> for an array of +link{AsyncOperationResult} objects,
// one for each of the <code>promises</code>.
Promise._whenAllSettled = function (promises, cancellationController) {
    var resolversObj = Promise.withResolvers();

    var results = new Array(promises.length);

    if (cancellationController && cancellationController.canceled) {
        Promise.__handleCancelWhenAllSettled(true, cancellationController, results, resolversObj);
        return resolversObj.promise;
    }

    Promise.allSettled(promises)
        .then(function (fulfillmentValues) {
            // If the givenCC is already canceled, then we do not want to modify the `results`
            // array, as it has already been (or will shortly be) given back to the caller.
            if (cancellationController && cancellationController.canceled) return;

            isc.Class._assert(results.length == fulfillmentValues.length);
            for (var i = 0; i < results.length; ++i) {
                var fulfillmentValue = fulfillmentValues[i];
                results[i] = fulfillmentValue.status == "fulfilled"
                             ? fulfillmentValue.value
                             : isc.defaultAsyncOperationCatchCallback(fulfillmentValue.reason);
            }

            resolversObj.resolve(results);
        });

    if (!cancellationController) return resolversObj.promise;

    

    var cancelObservationId = isc.observe(cancellationController, "cancel",
            "Promise.__handleCancelWhenAllSettled(returnVal,observed,observer._results,observer._resolversObj)");
    isc.addPropertiesWithAssign(isc.Page._staticObservers[cancelObservationId].observer, {
        _resolversObj: resolversObj,
        _results: results
    });

    return resolversObj.promise
        ._finally(function () {
            isc.ignore(cancelObservationId);
        });
};

Promise.__handleCancelWhenAllSettled = function (wasFirstCancel, givenCC, results, resolversObj) {
    if (!wasFirstCancel) return;
    var canceledResult = givenCC.asCanceledResult();
    for (var i = 0; i < results.length; ++i) {
        if (!results[i]) results[i] = canceledResult;
    }
    resolversObj.resolve(results);
};

//> @class ResolvedValuesMapper
// A utility class that coordinates mapping resolved values.
//<
isc.defineClass("ResolvedValuesMapper").addProperties({

    //> @attr ResolvedValuesMapper.values (Array of Any : [] : IR)
    // Values to be resolved. These are resolved in the same way as Promise.resolve().
    // In the case of rejection or a provided +link{ResolvedValuesMapper.cancellationController}
    // is canceled, then +link{ResolvedValuesMapper.asyncMap()} will not be called.
    //<
    values: [],

    //> @attr ResolvedValuesMapper.keepSettledStatus (Boolean : false : IR)
    // If <code>true</code>, then each fulfillment value will be an object having the following
    // properties:
    // <ul>
    // <li><code>status</code> - either "fulfilled" or "rejected"
    // <li><code>value</code> - only present if <code>status</code> is "fulfilled", the resolved
    // mapped value.
    // <li><code>reason</code> - only present if <code>status</code> is "rejected", the reason
    // why resolving the value rejected, or the reason why the mapping of the resolved value rejected.
    // </ul>
    // Note: This is the same fulfillment value format as Promise.allSettled().
    // <p>
    // If <code>false</code>, then the <code>Promise</code> returned by
    // +link{ResolvedValuesMapper.getPromise()}&mdash;if it resolves rather than rejects&mdash;will
    // resolve to an array of the resolved mapped values.
    // <p>
    // If <code>null</code>, then nothing about the resolution of each value or its mapped value
    // is retained. This is similar to an asynchronous forEach().
    //<
    keepSettledStatus: false,

    //> @attr ResolvedValuesMapper.cancellationController (CancellationController : null : IR)
    //<
    cancellationController: null,

    //> @attr ResolvedValuesMapper.maxConcurrent (Integer : 10 : IR)
    // Maximum number of concurrent invocations of +link{ResolvedValuesMapper.asyncMap()}
    // at any given time.
    // <p>
    // Set <code>maxConcurrent</code> to 1 to guarantee resolution of the mapped value of each
    // resolved value, one at a time in sequence.
    //<
    maxConcurrent: 10,

    init : function () {
        this.Super("init", arguments);

        var origMaxConcurrent = this.maxConcurrent,
            maxConcurrent = +origMaxConcurrent;
        if (!isc.isA.Number(maxConcurrent) || maxConcurrent <= 0) {
            var defaultMaxConcurrent = this.getClass().getInstanceProperty("maxConcurrent");
            if (origMaxConcurrent != null) this.logWarn("Invalid maxConcurrent. Defaulting to " + defaultMaxConcurrent);
            maxConcurrent = defaultMaxConcurrent;
        }
        this.maxConcurrent = maxConcurrent;

        this._resolversObj = Promise.withResolvers();
        if (this.keepSettledStatus != null) this._mappedFulfillmentValues = [];
        this._index = 0;
        this._numPendingMapInvocations = 0;

        // Even if a provided CancellationController is already canceled, we still need to do
        // either allSettled() or all() on `this.values` because resolution of one value might
        // reject, and that would be an uncaught rejection.
        if (this.keepSettledStatus == true) {
            Promise.allSettled(this.values)
                .then(this._continueWithFulfillmentValues.bind(this));
        } else {
            Promise.all(this.values)
                .then(
                    this._continueWithResolvedValues.bind(this),
                    this._handleReject.bind(this, null)
                );
        }

        var cancellationController = this.cancellationController;
        if (cancellationController) {
            if (cancellationController.canceled) {
                this.__handleCancel(true);
            } else {
                this.observe(cancellationController, "cancel", "observer.__handleCancel(returnVal)");
            }
        }
    },

    _continueWithFulfillmentValues : function (fulfillmentValues) {
        this._assert(this.keepSettledStatus == true);
        this._assert(fulfillmentValues.length == this.values.length);

        // If we're stopped, then we don't want to modify `this._mappedFulfillmentValues`
        // because this array has already been handed off to the caller.
        if (!this._stopped) {
            // Scan through `fulfillmentValues`, looking for rejects. Any such rejected
            // fulfillment value will be the mapped fulfillment value.
            for (var i = 0, len = this.values.length; i < len; ++i) {
                if (fulfillmentValues[i].status == "rejected") {
                    this._mappedFulfillmentValues[i] = fulfillmentValues[i];
                }
            }
        }

        this._fulfillmentValues = fulfillmentValues;
        this._continue();
    },

    _continueWithResolvedValues : function (values) {
        this._assert(this.keepSettledStatus != true);
        this._assert(values.length == this.values.length);

        this._values = values;
        this._continue();
    },

    _continue : function () {
        if (this._stopped) {
            return;
        }

        var fulfillmentValues,
            values;
        if (this.keepSettledStatus == true) {
            fulfillmentValues = this._fulfillmentValues;
        } else {
            values = this._values;
        }

        var maxConcurrent = this.maxConcurrent,
            numPendingMapInvocations = this._numPendingMapInvocations;
        for (var i = this._index, len = this.values.length;
             numPendingMapInvocations < maxConcurrent && i < len;
             i = ++this._index)
        {
            var value;
            if (this.keepSettledStatus == true) {
                var fulfillmentValue = fulfillmentValues[i];
                if (fulfillmentValue.status == "rejected") {
                    this._assert(this._mappedFulfillmentValues[i] === fulfillmentValue);
                    continue;
                }
                this._assert("value" in fulfillmentValue);
                value = fulfillmentValue.value;
            } else {
                value = values[i];
            }

            this.asyncMap(value, i)
                .then(
                    this._handleResolve.bind(this, i),
                    this._handleReject.bind(this, i)
                );
            numPendingMapInvocations = ++this._numPendingMapInvocations;
        }

        // It is possible for an asyncMap() call to cancel the provided CancellationController,
        // causing us to stop.

        this._assert(numPendingMapInvocations == this._numPendingMapInvocations);
        this._assert(i == this._index);

        if (numPendingMapInvocations <= 0 && i >= this.values.length) {
            this._resolversObj.resolve(this._mappedFulfillmentValues);
            this._cleanupOnStopped();
        }
    },

    _handleResolve : function (i, mappedValue) {
        this._assert(i != null);
        this._assert(0 <= i && i < this.values.length);
        this._assert(this._numPendingMapInvocations >= 1);
        --this._numPendingMapInvocations;

        if (this._stopped) return;

        if (this.keepSettledStatus == true) {
            this._mappedFulfillmentValues[i] = {
                status: "fulfilled",
                value: mappedValue
            };
        } else if (this.keepSettledStatus == false) {
            this._mappedFulfillmentValues[i] = mappedValue;
        } else {
            // we're ignoring the mapped values
        }

        this._continue();
    },

    _handleReject : function (i, rejectReason) {
        if (i != null) {
            this._assert(0 <= i && i < this.values.length);
            this._assert(this._numPendingMapInvocations >= 1);
            --this._numPendingMapInvocations;
        }

        if (this._stopped) return;

        if (this.keepSettledStatus == true) {
            this._assert(i != null);
            this._mappedFulfillmentValues[i] = {
                status: "rejected",
                reason: rejectReason
            };
        } else {
            this._resolversObj.reject(rejectReason);
            this._cleanupOnStopped();
            return;
        }

        this._continue();
    },

    _cleanupOnStopped : function () {
        this._assert(!this._stopped);
        this._stopped = true;
        var cancellationController = this.cancellationController;
        if (cancellationController && this.isObserving(cancellationController, "cancel")) {
            this.ignore(cancellationController, "cancel");
        }
    },

    __handleCancel : function (wasFirstCancel) {
        if (!wasFirstCancel) return;
        var cancellationController = this.cancellationController,
            canceledResult = cancellationController.asCanceledResult();
        if (this.keepSettledStatus == true) {
            var mappedFulfillmentValues = this._mappedFulfillmentValues;
            for (var i = 0, len = this.values.length; i < len; ++i) {
                if (mappedFulfillmentValues[i] == null) {
                    mappedFulfillmentValues[i] = {
                        status: "rejected",
                        reason: canceledResult
                    };
                }
            }

            this._resolversObj.resolve(mappedFulfillmentValues);
        } else {
            this._resolversObj.reject(canceledResult);
        }
        this._cleanupOnStopped();
    },

    getPromise : function () {
        return this._resolversObj.promise;
    },

    //> @method ResolvedValuesMapper.asyncMap()
    // Overridable method to map the resolved value at index <code>index</code>. The default
    // implementation is identity.
    // <p>
    // Note: Calls to this method may be skipped, even if +link{keepSettledStatus} is <code>true</code>,
    // if the provided +link{ResolvedValuesMapper.cancellationController} is canceled.
    // @param value (Any) the resolved value at index <code>index</code>
    // @param index (Integer) index
    // @return (Promise) a <code>Promise</code> for the mapped value.
    //<
    asyncMap : function (value, index) {
        return Promise.resolve(value);
    }
});


