//> TestStatusReport
// Test regression status report module that analyzes test results from the AutoTestResults
// database to identify tests broken by developer commits.
//
// Works in both browser (SmartClient UI) and Node.js (command-line) contexts.
//
// ====================================================================================
// FILTERING APPROACH DOCUMENTATION
// ====================================================================================
//
// This module uses multiple levels of filtering to ensure only actionable regressions
// are reported, excluding noise that would obscure real problems:
//
// 1. SYSTEM-WIDE FAILURE FILTERING (batches with >50 new failures)
//    - If a single commit causes >50 tests to break, this is almost certainly a
//      system-wide infrastructure issue (server crash, disk full, network outage,
//      database corruption) rather than a code bug.
//    - These are filtered at the SQL level via: HAVING COUNT(*) < 50
//    - Rationale: A developer commit rarely breaks more than a handful of related
//      tests. Mass failures indicate environmental problems.
//
// 2. PRE-EXISTING WOBBLER EXCLUSION (tests with >=25% fail rate before the commit)
//    - A "wobbler" is a test that intermittently passes/fails without code changes.
//    - If a test already had >=25% failure rate before a commit, blaming that commit
//      for subsequent failures is unfair and misleading.
//    - Filtered via: failsBeforeCommit * 100 / (passesBeforeCommit + failsBeforeCommit) < 25
//    - Rationale: The test was already unreliable; the commit didn't cause the problem.
//
// 3. MINIMUM FAILURE THRESHOLD (tests with <6 failures since the commit are ignored)
//    - Very few failures could be transient infrastructure issues rather than real bugs.
//    - Filtered via: failsSince >= 6
//    - Rationale: Ensures the pattern is established, not a one-off glitch.
//
// 4. LOOKBACK WINDOW (30 days before commit for pass/fail history)
//    - We look at 30 days of history before each commit to establish baseline behavior.
//    - This provides enough data for statistical reliability while staying relevant.
//
// These filters work together to ensure the report shows real, actionable regressions
// that developers are responsible for fixing, not noise or pre-existing problems.
//
// ====================================================================================
//<

(function() {

// CVSweb base URL for generating commit links
var CVSWEB_BASE = "http://cvs.isomorphic.com/cgi-bin/cvsweb.cgi";

// ==================================================================================
// TestStatusReport - Main analysis module
// ==================================================================================

var TestStatusReport = {

    // Default configuration
    config: {
        defaultDateRangeMonths: 2,
        branch: "MAIN",
        minFailures: 6,
        maxBatchImpact: 50,        // Batches with more failures are system-wide issues
        preExistingWobblerThreshold: 25  // % fail rate that makes it a pre-existing wobbler
    },

    // ================================================================================
    // Data fetching methods - work in both browser and Node.js
    // ================================================================================

    // Fetch new tests by a user that never worked
    // These are tests the user added that have never passed since creation.
    fetchNewTestsNeverWorked : function(userName, startDate, endDate, callback) {
        var criteria = {
            userName: userName,
            startDate: this._formatDate(startDate),
            endDate: this._formatDate(endDate)
        };
        this._fetch("autoTestAnalysis", "newTestsNeverWorked", criteria, callback);
    },

    // Fetch tests broken by a user's commits
    // Identifies tests that were working before the commit and broke after.
    // Automatically filters out:
    //   - System-wide failures (batches with >50 new failures)
    //   - Pre-existing wobblers (tests with >=25% fail rate before commit)
    fetchBrokenByUser : function(userName, startDate, endDate, callback) {
        var criteria = {
            userName: userName,
            startDate: this._formatDate(startDate),
            endDate: this._formatDate(endDate)
        };
        this._fetch("autoTestAnalysis", "brokenByUser", criteria, function(data, error) {
            if (error) {
                callback(null, error);
                return;
            }
            // Separate into broken vs wobbler categories
            var broken = [];
            var wobblers = [];
            data.forEach(function(record) {
                if (record.category === "BROKE_AND_STAYED_BROKEN") {
                    broken.push(record);
                } else if (record.category === "BECAME_WOBBLER") {
                    wobblers.push(record);
                }
                // RECOVERED_OR_PREEXISTING are intentionally excluded
            });
            callback({ broken: broken, wobblers: wobblers });
        });
    },

    // Fetch summary statistics
    fetchSummary : function(userName, startDate, endDate, callback) {
        var criteria = {
            userName: userName,
            startDate: this._formatDate(startDate),
            endDate: this._formatDate(endDate)
        };
        this._fetch("autoTestAnalysis", "summary", criteria, callback);
    },

    // ================================================================================
    // High-level analysis methods
    // ================================================================================

    // Get complete analysis for a single developer
    // Returns: { userName, startDate, endDate, newTestsNeverWorked, brokenTests, wobblers,
    //            filteringApplied, cvswebLinks }
    analyzeUser : function(userName, startDate, endDate, callback) {
        var self = this;
        var result = {
            userName: userName,
            startDate: this._formatDate(startDate),
            endDate: this._formatDate(endDate),
            newTestsNeverWorked: [],
            brokenTests: [],
            wobblers: [],
            filteringApplied: this._getFilteringDescription(),
            cvswebLinks: {}
        };

        // Fetch both data sets in parallel
        var pending = 2;
        var errors = [];

        function checkDone() {
            pending--;
            if (pending === 0) {
                if (errors.length > 0) {
                    callback(null, errors.join("; "));
                } else {
                    // Add CVSweb links
                    self._addCvswebLinks(result);
                    callback(result);
                }
            }
        }

        // Fetch new tests that never worked
        this.fetchNewTestsNeverWorked(userName, startDate, endDate, function(data, error) {
            if (error) {
                errors.push("newTests: " + error);
            } else {
                result.newTestsNeverWorked = data || [];
            }
            checkDone();
        });

        // Fetch broken tests
        this.fetchBrokenByUser(userName, startDate, endDate, function(data, error) {
            if (error) {
                errors.push("brokenTests: " + error);
            } else if (data) {
                result.brokenTests = data.broken || [];
                result.wobblers = data.wobblers || [];
            }
            checkDone();
        });
    },

    // Analyze multiple users and combine results
    analyzeUsers : function(userNames, startDate, endDate, callback) {
        var self = this;
        var results = [];
        var pending = userNames.length;
        var errors = [];

        userNames.forEach(function(userName) {
            self.analyzeUser(userName, startDate, endDate, function(result, error) {
                if (error) {
                    errors.push(userName + ": " + error);
                } else {
                    results.push(result);
                }
                pending--;
                if (pending === 0) {
                    // Sort by total issues (descending)
                    results.sort(function(a, b) {
                        var aTotal = a.newTestsNeverWorked.length + a.brokenTests.length + a.wobblers.length;
                        var bTotal = b.newTestsNeverWorked.length + b.brokenTests.length + b.wobblers.length;
                        return bTotal - aTotal;
                    });
                    callback({
                        results: results,
                        errors: errors.length > 0 ? errors : null
                    });
                }
            });
        });
    },

    // Get top N commits with most test breakage
    getTopBadCommits : function(startDate, endDate, topN, callback) {
        topN = topN || 5;
        var self = this;

        // First fetch all batches in the date range
        var criteria = {
            branch: this.config.branch,
            batchStartTime: { greaterThan: this._formatDate(startDate), lessThan: this._formatDate(endDate) },
            completionStatus: "success"
        };

        this._fetch("sourceCommit", "fetch", criteria, function(batches, error) {
            if (error) {
                callback(null, error);
                return;
            }

            // For each batch, count new failures (tests that passed before and failed after)
            // This requires additional queries - simplified version uses batch impact from analysis
            var commitImpacts = [];
            batches.forEach(function(batch) {
                commitImpacts.push({
                    batchId: batch.id,
                    batchStartTime: batch.batchStartTime,
                    user: batch.user,
                    log: batch.log,
                    modifiedFiles: batch.modifiedFiles,
                    cvswebLink: self._makeCvswebBatchLink(batch),
                    // Impact will be populated from brokenByUser results
                    testsImpacted: 0
                });
            });

            // For now, return what we have - actual impact calculation would need
            // cross-referencing with brokenByUser results for each batch
            commitImpacts.sort(function(a, b) { return b.testsImpacted - a.testsImpacted; });
            callback(commitImpacts.slice(0, topN));
        });
    },

    // ================================================================================
    // Output formatting methods
    // ================================================================================

    // Format as ASCII table for command-line output
    toAscii : function(analysisResult) {
        var lines = [];
        var sep = "=".repeat(80);
        var thinSep = "-".repeat(80);

        lines.push(sep);
        lines.push("TEST STATUS REPORT: " + analysisResult.userName);
        lines.push("Date Range: " + analysisResult.startDate + " to " + analysisResult.endDate);
        lines.push(sep);
        lines.push("");

        // Summary
        var totalIssues = analysisResult.newTestsNeverWorked.length +
                          analysisResult.brokenTests.length +
                          analysisResult.wobblers.length;

        if (totalIssues === 0) {
            lines.push("  [ALL CLEAR] No test regressions found for this user/period.");
        } else {
            lines.push("  ISSUES FOUND: " + totalIssues);
            lines.push("    - New tests that never worked: " + analysisResult.newTestsNeverWorked.length);
            lines.push("    - Pre-existing tests broken:   " + analysisResult.brokenTests.length);
            lines.push("    - Tests that became wobblers:  " + analysisResult.wobblers.length);
        }
        lines.push("");

        // Filtering applied
        lines.push(thinSep);
        lines.push("FILTERING APPLIED:");
        lines.push(thinSep);
        analysisResult.filteringApplied.forEach(function(f) {
            lines.push("  - " + f);
        });
        lines.push("");

        // New tests that never worked
        if (analysisResult.newTestsNeverWorked.length > 0) {
            lines.push(thinSep);
            lines.push("NEW TESTS THAT NEVER WORKED (" + analysisResult.newTestsNeverWorked.length + "):");
            lines.push(thinSep);
            analysisResult.newTestsNeverWorked.forEach(function(t) {
                lines.push("  " + t.testFile);
                lines.push("    First seen: " + t.firstSeen + " | Fails: " + t.failsSince);
                if (t.cvswebLink) lines.push("    CVSweb: " + t.cvswebLink);
            });
            lines.push("");
        }

        // Broken tests
        if (analysisResult.brokenTests.length > 0) {
            lines.push(thinSep);
            lines.push("PRE-EXISTING TESTS BROKEN (" + analysisResult.brokenTests.length + "):");
            lines.push(thinSep);
            analysisResult.brokenTests.forEach(function(t) {
                lines.push("  " + t.testFile);
                lines.push("    Broke: " + t.brokeOnDate + " | Before: " + t.passesBeforeCommit +
                           " pass | Since: " + t.failsSince + " fail");
                if (t.cvswebLink) lines.push("    CVSweb: " + t.cvswebLink);
            });
            lines.push("");
        }

        // Wobblers
        if (analysisResult.wobblers.length > 0) {
            lines.push(thinSep);
            lines.push("TESTS THAT BECAME WOBBLERS (" + analysisResult.wobblers.length + "):");
            lines.push(thinSep);
            analysisResult.wobblers.forEach(function(t) {
                lines.push("  " + t.testFile);
                lines.push("    Since: " + t.brokeOnDate + " | Fail rate: " + t.failPct +
                           "% | Before: " + t.passesBeforeCommit + "/" +
                           (t.passesBeforeCommit + t.failsBeforeCommit) + " pass");
                if (t.cvswebLink) lines.push("    CVSweb: " + t.cvswebLink);
            });
            lines.push("");
        }

        lines.push(sep);

        return lines.join("\n");
    },

    // Format multiple user results as ASCII
    toAsciiMultiUser : function(multiUserResult) {
        var lines = [];
        var sep = "=".repeat(80);

        lines.push(sep);
        lines.push("TEST STATUS REPORT - MULTIPLE USERS");
        lines.push(sep);
        lines.push("");

        // Summary table
        lines.push("USER                  NEW_BROKEN  PRE_BROKEN  WOBBLERS  TOTAL");
        lines.push("-".repeat(70));

        var grandTotal = { newBroken: 0, preBroken: 0, wobblers: 0 };

        multiUserResult.results.forEach(function(r) {
            var newB = r.newTestsNeverWorked.length;
            var preB = r.brokenTests.length;
            var wobb = r.wobblers.length;
            var total = newB + preB + wobb;

            grandTotal.newBroken += newB;
            grandTotal.preBroken += preB;
            grandTotal.wobblers += wobb;

            var userName = (r.userName + "                    ").slice(0, 20);
            lines.push(userName + "  " +
                       String(newB).padStart(10) + "  " +
                       String(preB).padStart(10) + "  " +
                       String(wobb).padStart(8) + "  " +
                       String(total).padStart(5));
        });

        lines.push("-".repeat(70));
        lines.push("TOTAL                 " +
                   String(grandTotal.newBroken).padStart(10) + "  " +
                   String(grandTotal.preBroken).padStart(10) + "  " +
                   String(grandTotal.wobblers).padStart(8) + "  " +
                   String(grandTotal.newBroken + grandTotal.preBroken + grandTotal.wobblers).padStart(5));
        lines.push("");

        // Individual user details
        multiUserResult.results.forEach(function(r) {
            lines.push("");
            lines.push(this.toAscii(r));
        }, this);

        if (multiUserResult.errors) {
            lines.push("");
            lines.push("ERRORS:");
            multiUserResult.errors.forEach(function(e) {
                lines.push("  " + e);
            });
        }

        return lines.join("\n");
    },

    // ================================================================================
    // Helper methods
    // ================================================================================

    _formatDate : function(date) {
        if (typeof date === "string") return date;
        var d = date instanceof Date ? date : new Date(date);
        return d.toISOString().slice(0, 10);
    },

    _getDefaultStartDate : function() {
        var d = new Date();
        d.setMonth(d.getMonth() - this.config.defaultDateRangeMonths);
        return this._formatDate(d);
    },

    _getDefaultEndDate : function() {
        return this._formatDate(new Date());
    },

    _getFilteringDescription : function() {
        return [
            "System-wide failures excluded (batches with >" + this.config.maxBatchImpact + " new failures)",
            "Pre-existing wobblers excluded (tests with >=" + this.config.preExistingWobblerThreshold + "% fail rate before commit)",
            "Minimum failure threshold: " + this.config.minFailures + " failures since commit",
            "History lookback: 30 days before each commit"
        ];
    },

    _addCvswebLinks : function(result) {
        var self = this;

        // Add links for new tests that never worked
        result.newTestsNeverWorked.forEach(function(t) {
            if (t.brokeInBatch) {
                t.cvswebLink = self._makeCvswebCommitLink(t.brokeInBatch);
            }
        });

        // Add links for broken tests
        result.brokenTests.forEach(function(t) {
            if (t.brokeInBatch) {
                t.cvswebLink = self._makeCvswebCommitLink(t.brokeInBatch);
            }
        });

        // Add links for wobblers
        result.wobblers.forEach(function(t) {
            if (t.brokeInBatch) {
                t.cvswebLink = self._makeCvswebCommitLink(t.brokeInBatch);
            }
        });
    },

    _makeCvswebCommitLink : function(batchId) {
        // Link to the batch/commit details - format depends on CVSweb setup
        return CVSWEB_BASE + "?batchId=" + batchId;
    },

    _makeCvswebBatchLink : function(batch) {
        return CVSWEB_BASE + "?batchId=" + batch.id;
    },

    // ================================================================================
    // Data fetch abstraction - works in both browser and Node.js
    // ================================================================================

    _fetch : function(dsName, operationId, criteria, callback) {
        // Browser context - use SmartClient DataSource
        if (typeof isc !== "undefined" && isc.DataSource) {
            var ds = isc.DataSource.get(dsName);
            if (!ds) {
                callback(null, "DataSource not found: " + dsName);
                return;
            }
            ds.fetchData(criteria, function(dsResponse, data, dsRequest) {
                if (dsResponse.status < 0) {
                    callback(null, dsResponse.data || "Fetch failed");
                } else {
                    callback(data);
                }
            }, { operationId: operationId });
        }
        // Node.js context - use direct database access
        else if (typeof require !== "undefined") {
            this._nodeFetch(dsName, operationId, criteria, callback);
        }
        else {
            callback(null, "No data fetch mechanism available");
        }
    },

    // Node.js database access
    _nodeFetch : function(dsName, operationId, criteria, callback) {
        // This will be called from the Node.js wrapper which handles DB connection
        if (this._nodeCallback) {
            this._nodeCallback(dsName, operationId, criteria, callback);
        } else {
            callback(null, "Node.js fetch not configured - use setNodeFetchCallback()");
        }
    },

    // Set callback for Node.js database fetching
    setNodeFetchCallback : function(callback) {
        this._nodeCallback = callback;
    }
};

// ==================================================================================
// Module export for both browser and Node.js
// ==================================================================================

if (typeof module !== "undefined" && module.exports) {
    // Node.js
    module.exports = TestStatusReport;
} else if (typeof isc !== "undefined") {
    // SmartClient browser context
    isc.TestStatusReport = TestStatusReport;
} else {
    // Generic browser
    window.TestStatusReport = TestStatusReport;
}

})();
