/*

  SmartClient Ajax RIA system
  Version SNAPSHOT_v15.0d_2026-02-25/LGPL Deployment (2026-02-25)

  Copyright 2000 and beyond Isomorphic Software, Inc. All rights reserved.
  "SmartClient" is a trademark of Isomorphic Software, Inc.

  LICENSE NOTICE
     INSTALLATION OR USE OF THIS SOFTWARE INDICATES YOUR ACCEPTANCE OF
     ISOMORPHIC SOFTWARE LICENSE TERMS. If you have received this file
     without an accompanying Isomorphic Software license file, please
     contact licensing@isomorphic.com for details. Unauthorized copying and
     use of this software is a violation of international copyright law.

  DEVELOPMENT ONLY - DO NOT DEPLOY
     This software is provided for evaluation, training, and development
     purposes only. It may include supplementary components that are not
     licensed for deployment. The separate DEPLOY package for this release
     contains SmartClient components that are licensed for deployment.

  PROPRIETARY & PROTECTED MATERIAL
     This software contains proprietary materials that are protected by
     contract and intellectual property law. You are expressly prohibited
     from attempting to reverse engineer this software or modify this
     software for human readability.

  CONTACT ISOMORPHIC
     For more information regarding license rights and restrictions, or to
     report possible license violations, please contact Isomorphic Software
     by email (licensing@isomorphic.com) or web (www.isomorphic.com).

*/
//> @class FileUploadItem
// FormItem for uploading files using HTML5 file drop capabilities.
// <P>
// A FileUploadItem's +link{fileUploadItem.canvas,canvas} is a +link{FileDropZone}.
// The +link{getValue(),value} of a FileUploadItem is the JavaScript File object(s)
// representing files the user added.
// <P>
// FileUploadItem is not the default for <code>binary</code> type fields.
// To use FileUploadItem for binary fields, set +link{dynamicForm.useFileUploadItem}
// to true on the form, or specify <code>editorType:"FileUploadItem"</code> on the field.
// <P>
// When the form is +link{dynamicForm.saveData(),saved}, files are uploaded automatically.
// Use +link{dynamicForm.showUploadProgress} to display upload progress.
// <P>
// <b>Single File Upload</b>
// <P>
// With the default setting of +link{multiple,multiple:false}, a FileUploadItem allows
// uploading a single file to a binary field on the DataSource. This is analogous to
// the standard +link{FileItem}.
// <P>
// <b>Multiple File Upload</b>
// <P>
// To upload multiple files, set +link{multiple,multiple:true} and configure a
// +link{dataSource} property pointing to a related DataSource that will store the files.
// This follows the same master-detail pattern as +link{MultiFileItem}:
// <ul>
// <li>The form's DataSource is the "master" record (e.g., an email message)</li>
// <li>The FileUploadItem's +link{dataSource} is the "detail" DataSource storing files
//     (e.g., email attachments)</li>
// <li>The detail DataSource must have a +link{dataSourceField.foreignKey,foreignKey}
//     linking to the master DataSource's primary key</li>
// <li>Each uploaded file creates a separate record in the detail DataSource</li>
// </ul>
// <P>
// See +link{MultiFileItem} for an example of the DataSource setup required.
// <P>
// <b>Multiple Binary Fields</b>
// <P>
// A form can contain multiple FileUploadItems for different binary fields in the same
// DataSource. When the form is saved, all files are uploaded in a single request,
// creating one record with all binary fields populated. This differs from +link{FileItem}
// which has a limitation preventing multiple file uploads in a single form submission.
// <P>
// Note: If you want immediate upload on drop (like Gmail attachments), use a
// +link{formItem.changed()} handler to call +link{dynamicForm.saveData()}.
//
// @inheritsFrom CanvasItem
// @treeLocation Client Reference/Forms/Form Items
// @visibility external
//<
isc.ClassFactory.defineClass("FileUploadItem", "CanvasItem");

isc.FileUploadItem.addProperties({

    // Allow focus to go to the FileDropZone
    canFocus: true,

    // The value of this item should be saved with the form
    shouldSaveValue: true,

    // Default height for the drop zone
    height: 150,

    //> @attr fileUploadItem.canvas (AutoChild FileDropZone : null : IR)
    // The +link{FileDropZone} component that provides the drag-drop file selection UI.
    // <P>
    // The FileDropZone is automatically created using the +link{canvasConstructor} and
    // +link{canvasDefaults} properties. To customize the FileDropZone, either set
    // properties directly on the FileUploadItem (which are passed through to the
    // FileDropZone) or override +link{canvasDefaults}.
    // <P>
    // Access the FileDropZone via <code>fileUploadItem.canvas</code> after the item
    // is created.
    // @visibility external
    //<

    //> @attr fileUploadItem.canvasConstructor (String : "FileDropZone" : IR)
    // The class to use for the +link{canvas} autoChild. Default is FileDropZone.
    // @visibility external
    //<
    canvasConstructor: "FileDropZone",

    //> @attr fileUploadItem.canvasDefaults (Object : {...} : IR)
    // Default properties for the +link{canvas} autoChild FileDropZone.
    // <P>
    // The default implementation includes handlers that forward +link{FileDropZone.filesAdded}
    // and +link{FileDropZone.filesRemoved} notifications to update the FormItem's value.
    // <P>
    // When overriding, use <code>isc.addProperties()</code> to merge with the defaults
    // rather than replacing them entirely, to preserve the value synchronization behavior.
    // @visibility external
    //<
    canvasDefaults: {
        // Forward filesAdded notification to the FormItem
        filesAdded : function (files) {
            var item = this.canvasItem;
            if (!files || files.length == 0) {
                item.storeValue(null);
            } else if (item.multiple) {
                item.storeValue(files);
            } else {
                item.storeValue(files[0]);
            }
        },

        // Forward filesRemoved notification
        filesRemoved : function (files) {
            var item = this.canvasItem;
            var currentFiles = this.getFiles();
            if (!currentFiles || currentFiles.length == 0) {
                item.storeValue(null);
            } else if (item.multiple) {
                item.storeValue(currentFiles);
            } else {
                item.storeValue(currentFiles[0]);
            }
        },

        // Forward processingCancelled notification to the form to abort the upload
        processingCancelled : function () {
            var item = this.canvasItem;
            if (item && item.form && item.form._cancelUploadProcessing) {
                item.form._cancelUploadProcessing();
            }
        }
    },

    // Properties to pass through to FileDropZone

    //> @attr fileUploadItem.multiple (Boolean : null : IR)
    // Whether this FileUploadItem allows multiple files to be selected.
    // <P>
    // When <code>multiple:true</code>, a +link{dataSource} property must also be specified
    // pointing to a related DataSource that will store the uploaded files, following the
    // same master-detail pattern as +link{MultiFileItem}. Each file will be uploaded as
    // a separate record in the detail DataSource after the master record is saved.
    // <P>
    // If <code>multiple:true</code> is set without a valid +link{dataSource}, a warning
    // will be logged and the item will behave as if <code>multiple:false</code>.
    // <P>
    // See +link{fileDropZone.multiple} for the underlying FileDropZone property.
    // @visibility external
    //<

    //> @attr fileUploadItem.dataSource (DataSource | ID : null : IR)
    // DataSource where files are stored when +link{multiple,multiple:true}.
    // <P>
    // This DataSource should contain:
    // <ul>
    // <li>A +link{dataSourceField.primaryKey,primaryKey} field</li>
    // <li>A field with a +link{dataSourceField.foreignKey,foreignKey} relationship to
    //     the primary key of the form's DataSource</li>
    // <li>A field of type "binary" for storing the uploaded file</li>
    // </ul>
    // <P>
    // This follows the same pattern as +link{MultiFileItem.dataSource}. See
    // +link{MultiFileItem} for a complete example of the required DataSource structure.
    // <P>
    // This property is required when +link{multiple,multiple:true} is set. If omitted,
    // a warning will be logged and the item will behave as single-file upload.
    // @visibility external
    //<

    //> @attr fileUploadItem.maxFiles (Integer : null : IR)
    // See +link{fileDropZone.maxFiles}.
    // @visibility external
    //<

    //> @attr fileUploadItem.maxSize (Integer : null : IR)
    // See +link{fileDropZone.maxSize}.
    // @visibility external
    //<

    //> @attr fileUploadItem.minSize (Integer : null : IR)
    // See +link{fileDropZone.minSize}.
    // @visibility external
    //<

    //> @attr fileUploadItem.maxFileSize (Integer : null : IR)
    // See +link{fileDropZone.maxFileSize}.
    // @visibility external
    //<

    //> @attr fileUploadItem.replaceFilesOnDrop (Boolean : null : IR)
    // See +link{fileDropZone.replaceFilesOnDrop}.
    // @visibility external
    //<

    //> @attr fileUploadItem.acceptedFileTypes (Array of String : null : IR)
    // See +link{fileDropZone.acceptedFileTypes}.
    // @visibility external
    //<

    //> @attr fileUploadItem.canAddFilesOnClick (Boolean : null : IR)
    // See +link{fileDropZone.canAddFilesOnClick}.
    // @visibility external
    //<

    //> @attr fileUploadItem.showFileThumbnails (Boolean : null : IR)
    // See +link{fileDropZone.showFileThumbnails}.
    // @visibility external
    //<

    //> @attr fileUploadItem.showImagePreviews (Boolean : null : IR)
    // See +link{fileDropZone.showImagePreviews}.
    // @visibility external
    //<

    //> @attr fileUploadItem.thumbnailWidth (Integer : null : IR)
    // See +link{fileDropZone.thumbnailWidth}.
    // @visibility external
    //<

    //> @attr fileUploadItem.thumbnailHeight (Integer : null : IR)
    // See +link{fileDropZone.thumbnailHeight}.
    // @visibility external
    //<

    //> @attr fileUploadItem.showCancelButton (Boolean : null : IR)
    // See +link{fileDropZone.showCancelButton}.
    // @visibility external
    //<

    // i18n message properties - passthrough to FileDropZone

    //> @attr fileUploadItem.emptyDropAreaMessage (String : null : IR)
    // Custom +link{fileDropZone.emptyDropAreaMessage} for this fileUploadItem's canvas.
    // If unset, the default property will be used.
    // @group i18nMessages
    // @visibility external
    //<

    //> @attr fileUploadItem.clickToAddMessage (String : null : IR)
    // Custom +link{fileDropZone.clickToAddMessage} for this fileUploadItem's canvas.
    // If unset, the default property will be used.
    // @group i18nMessages
    // @visibility external
    //<

    //> @attr fileUploadItem.multipleFilesErrorMessage (String : null : IR)
    // Custom +link{fileDropZone.multipleFilesErrorMessage} for this fileUploadItem's canvas.
    // If unset, the default property will be used.
    // @group i18nMessages
    // @visibility external
    //<

    //> @attr fileUploadItem.maxFilesErrorMessage (String : null : IR)
    // Custom +link{fileDropZone.maxFilesErrorMessage} for this fileUploadItem's canvas.
    // If unset, the default property will be used.
    // @group i18nMessages
    // @visibility external
    //<

    //> @attr fileUploadItem.maxSizeErrorMessage (String : null : IR)
    // Custom +link{fileDropZone.maxSizeErrorMessage} for this fileUploadItem's canvas.
    // If unset, the default property will be used.
    // @group i18nMessages
    // @visibility external
    //<

    //> @attr fileUploadItem.minSizeErrorMessage (String : null : IR)
    // Custom +link{fileDropZone.minSizeErrorMessage} for this fileUploadItem's canvas.
    // If unset, the default property will be used.
    // @group i18nMessages
    // @visibility external
    //<

    //> @attr fileUploadItem.maxFileSizeErrorMessage (String : null : IR)
    // Custom +link{fileDropZone.maxFileSizeErrorMessage} for this fileUploadItem's canvas.
    // If unset, the default property will be used.
    // @group i18nMessages
    // @visibility external
    //<

    //> @attr fileUploadItem.duplicateFileNameMessage (String : null : IR)
    // Custom +link{fileDropZone.duplicateFileNameMessage} for this fileUploadItem's canvas.
    // If unset, the default property will be used.
    // @group i18nMessages
    // @visibility external
    //<

    //> @attr fileUploadItem.invalidFileTypeMessage (String : null : IR)
    // Custom +link{fileDropZone.invalidFileTypeMessage} for this fileUploadItem's canvas.
    // If unset, the default property will be used.
    // @group i18nMessages
    // @visibility external
    //<

    //> @attr fileUploadItem.processingMessage (String : null : IR)
    // Custom +link{fileDropZone.processingMessage} for this fileUploadItem's canvas.
    // If unset, the default property will be used.
    // @group i18nMessages
    // @visibility external
    //<

    //> @attr fileUploadItem.cancelButtonTitle (String : null : IR)
    // Custom +link{fileDropZone.cancelButtonTitle} for this fileUploadItem's canvas.
    // If unset, the default property will be used.
    // @group i18nMessages
    // @visibility external
    //<

    // List of properties to pass through to FileDropZone
    _passthroughProperties: [
        "multiple", "maxFiles", "maxSize", "minSize", "maxFileSize",
        "replaceFilesOnDrop", "acceptedFileTypes", "canAddFilesOnClick",
        "showFileThumbnails", "showImagePreviews", "thumbnailWidth", "thumbnailHeight",
        "showCancelButton",
        // i18n messages
        "emptyDropAreaMessage", "clickToAddMessage", "multipleFilesErrorMessage",
        "maxFilesErrorMessage", "maxSizeErrorMessage", "minSizeErrorMessage",
        "maxFileSizeErrorMessage", "duplicateFileNameMessage", "invalidFileTypeMessage",
        "processingMessage", "cancelButtonTitle"
    ]
});


isc.FileUploadItem.addMethods({

init : function () {
    this.Super("init", arguments);

    // Validate configuration: multiple:true requires a dataSource for storing files
    if (this.multiple && !this.dataSource) {
        this.logWarn("FileUploadItem with multiple:true requires a 'dataSource' " +
            "property specifying a related DataSource for storing uploaded files. " +
            "This follows the same master-detail pattern as MultiFileItem. " +
            "Without a dataSource, the item will behave as single-file upload. " +
            "See the FileUploadItem documentation for details.");
        // Fall back to single-file mode
        this.multiple = false;
    }

    // Track pending files for upload after master record save (multi-file mode)
    this._pendingFiles = [];
},

//> @method fileUploadItem.getDataSource()
// Returns the DataSource configured for storing uploaded files when
// +link{multiple,multiple:true}.
// @return (DataSource) the detail DataSource, or null if not configured
// @visibility external
//<
getDataSource : function () {
    if (this.dataSource == null) return null;
    return isc.DataSource.get(this.dataSource);
},

// Returns true if this item is configured for multi-file upload with a related DataSource
_isMultiFileMode : function () {
    return this.multiple && this.dataSource != null;
},

// Called by DynamicForm after the master record is saved. Uploads pending files to
// the related DataSource. Returns false to pause the save callback chain if there
// are files to upload; callback will be resumed after uploads complete.
formSaved : function (request, response, data) {
    if (!this._isMultiFileMode()) return true;

    var files = this.getFiles();
    if (!files || files.length == 0) return true;

    // Get the detail DataSource and determine the foreign key field
    var detailDS = this.getDataSource();
    if (!detailDS) return true;

    var formDS = this.form.getDataSource();
    if (!formDS) {
        this.logWarn("Cannot upload files: form has no DataSource");
        return true;
    }

    // Get foreign key values from the saved master record
    var foreignKeyValues = detailDS.getForeignKeysByRelation(data, formDS);
    if (isc.isAn.emptyObject(foreignKeyValues)) {
        this.logWarn("Cannot determine foreign key values for detail DataSource. " +
            "Ensure the detail DataSource has a foreignKey field referencing the " +
            "form DataSource's primary key.");
        return true;
    }

    // Find the binary field in the detail DataSource
    var binaryFieldName = this._getDetailBinaryFieldName(detailDS);
    if (!binaryFieldName) {
        this.logWarn("Detail DataSource '" + detailDS.getID() +
            "' has no binary field for storing files");
        return true;
    }

    // Start upload processing UI
    this.startProcessing();

    // Upload files sequentially to the detail DataSource
    var _this = this;
    var fileIndex = 0;
    var totalSize = 0;
    var uploadedSize = 0;

    for (var i = 0; i < files.length; i++) {
        totalSize += files[i].size;
    }

    var uploadNextFile = function () {
        if (fileIndex >= files.length) {
            // All files uploaded
            _this.endProcessing();
            _this.canvas.clearFiles();
            // Resume the form's save callback chain
            if (_this._formSavedCallback) {
                var callback = _this._formSavedCallback;
                delete _this._formSavedCallback;
                _this.form._fileUploadItemsSaved(callback);
            }
            return;
        }

        var file = files[fileIndex];
        var record = isc.addProperties({}, foreignKeyValues);
        record[binaryFieldName] = file;

        var fileStartSize = uploadedSize;

        detailDS.addData(record, function(dsResponse, dsData, dsRequest) {
            if (dsResponse.status == 0) {
                uploadedSize += file.size;
                fileIndex++;
                uploadNextFile();
            } else {
                _this.logWarn("Failed to upload file: " + file.name);
                _this.endProcessing();
            }
        }, {
            xhrUpload: true,
            xhrUploadProgress: function(loaded, total) {
                var overallLoaded = fileStartSize + loaded;
                var percent = totalSize > 0 ?
                    Math.round((overallLoaded / totalSize) * 100) : 0;
                _this.setProcessingProgress(percent, overallLoaded, totalSize);
            }
        });
    };

    // Store callback and start uploading
    this._formSavedCallback = request.afterFlowCallback;
    uploadNextFile();

    // Return false to pause the callback chain - we'll resume it after uploads complete
    return false;
},

// Finds the binary field in a DataSource
_getDetailBinaryFieldName : function (ds) {
    var fields = ds.getFields();
    for (var fieldName in fields) {
        if (fields[fieldName].type == "binary") {
            return fieldName;
        }
    }
    return null;
},

    // Override _createCanvas to pass through properties from FileUploadItem to the
    // FileDropZone autoChild. Properties listed in _passthroughProperties are copied
    // to the canvas config if set on the FileUploadItem, allowing developers to
    // configure FileDropZone behavior directly on the form field definition.
    _createCanvas : function () {
        var properties = {};

        // Pass through configured properties from FileUploadItem to FileDropZone
        for (var i = 0; i < this._passthroughProperties.length; i++) {
            var prop = this._passthroughProperties[i];
            if (this[prop] != null) {
                properties[prop] = this[prop];
            }
        }

        // Merge passthrough properties into the canvas config
        this.canvas = isc.addProperties({}, properties);

        // Call superclass implementation which creates the actual FileDropZone
        this.Super("_createCanvas", arguments);
    },

    // Override showValue to sync FileDropZone display with item value.
    // Only passes actual File/Blob objects to the FileDropZone - after a form save,
    // the dataValue may be server response data which is not a File object.
    // Preserves existing files if the new value isn't a valid Blob, since formSaved
    // may still need those files for upload. formSaved clears them after upload completes.
    showValue : function (displayValue, dataValue, form, item) {
        if (this.canvas) {
            if (dataValue == null) {
                this.canvas.clearFiles();
            } else {
                var files = isc.isAn.Array(dataValue) ? dataValue : [dataValue];
                // Filter to only include actual File/Blob objects - server response data
                // won't have these types
                var validFiles = [];
                for (var i = 0; i < files.length; i++) {
                    var file = files[i];
                    if (file && (file instanceof Blob)) {
                        validFiles.push(file);
                    }
                }
                if (validFiles.length > 0) {
                    this.canvas.setFiles(validFiles);
                }
                // If no valid Blob files, preserve existing files - they may be pending
                // upload via formSaved which will clear them after upload completes
            }
        }
    },

    //> @method fileUploadItem.startProcessing()
    // See +link{fileDropZone.startProcessing()}.
    // @visibility external
    //<
    startProcessing : function () {
        if (this.canvas) return this.canvas.startProcessing();
        return false;
    },

    //> @method fileUploadItem.setProcessingProgress()
    // See +link{fileDropZone.setProcessingProgress()}.
    // @visibility external
    //<
    setProcessingProgress : function (percentDone, processed, total) {
        if (this.canvas) {
            return this.canvas.setProcessingProgress(percentDone, processed, total);
        }
        return false;
    },

    //> @method fileUploadItem.setFileProgress()
    // See +link{fileDropZone.setFileProgress()}.
    // @visibility external
    //<
    setFileProgress : function (file, percentDone, processed, total) {
        if (this.canvas) {
            return this.canvas.setFileProgress(file, percentDone, processed, total);
        }
        return false;
    },

    //> @method fileUploadItem.endProcessing()
    // See +link{fileDropZone.endProcessing()}.
    // @visibility external
    //<
    endProcessing : function () {
        if (this.canvas) this.canvas.endProcessing();
    },

    //> @method fileUploadItem.cancelProcessing()
    // See +link{fileDropZone.cancelProcessing()}.
    // @visibility external
    //<
    cancelProcessing : function () {
        if (this.canvas) this.canvas.cancelProcessing();
    },

    //> @method fileUploadItem.getSize()
    // See +link{fileDropZone.getSize()}.
    // @visibility external
    //<
    getSize : function () {
        if (this.canvas) return this.canvas.getSize();
        return 0;
    },

    //> @method fileUploadItem.getFiles()
    // See +link{fileDropZone.getFiles()}.
    // @visibility external
    //<
    getFiles : function () {
        if (this.canvas) return this.canvas.getFiles();
        return [];
    }

});
