// FileUploadItem in DynamicForm - demonstrates HTML5 drag-drop upload with form integration
// This sample uses FileUploadItem for a binary field, enabling drag-and-drop file selection
// with thumbnail previews and progress indication during upload.

// Create a container for the current image preview (shown when editing existing records)
isc.Img.create({
    ID: "currentImagePreview",
    width: 100,
    height: 75,
    imageType: "stretch",
    backgroundColor: "#f5f5f5",
    showEdges: true,
    visibility: "hidden",
    src: "[SKINIMG]blank.gif"
});

isc.Label.create({
    ID: "currentImageLabel",
    height: 20,
    visibility: "hidden",
    contents: ""
});

isc.VLayout.create({
    ID: "currentImagePanel",
    width: 120,
    visibility: "hidden",
    members: [
        isc.Label.create({contents: "<b>Current Image:</b>", height: 20}),
        currentImagePreview,
        currentImageLabel
    ],
    membersMargin: 5
});

isc.DynamicForm.create({
    ID: "uploadForm",
    width: 350,
    dataSource: "fileDropZoneMedia",
    // Enable FileUploadItem for binary fields
    useFileUploadItem: true,
    fields: [
        {name: "title", title: "Title", required: true},
        {name: "image", title: "Image", type: "binary", height: 150,
         // FileUploadItem properties passed through to FileDropZone
         multiple: false,
         showFileThumbnails: true,
         showImagePreviews: true,
         emptyDropAreaMessage: "Drop an image here",
         clickToAddMessage: "or click to select"
        },
        {type: "button", title: "Upload", align: "center",
         click: function() {
             if (uploadForm.validate()) {
                 uploadForm.saveData(function(response, data) {
                     if (response.status === 0) {
                         isc.say("Upload successful!");
                         uploadForm.editNewRecord();
                         mediaGrid.invalidateCache();
                         // Hide current image panel since we're now editing new
                         currentImagePanel.hide();
                     }
                 });
             }
         }
        }
    ],
    // When editing a record, show the current image in a separate panel
    editRecord : function (record) {
        this.Super("editRecord", arguments);

        // Show existing image if the record has one
        if (record && record.image_filename) {
            var ds = isc.DataSource.get("fileDropZoneMedia");
            var imgURL = ds.getFileURL(record, "image");
            currentImagePreview.setSrc(imgURL);
            currentImageLabel.setContents(record.image_filename);
            currentImagePanel.show();
            currentImagePreview.show();
            currentImageLabel.show();
        } else {
            currentImagePanel.hide();
        }
    },
    editNewRecord : function () {
        this.Super("editNewRecord", arguments);
        currentImagePanel.hide();
    }
});

// Grid showing uploaded media
isc.ListGrid.create({
    ID: "mediaGrid",
    width: 400,
    height: 200,
    dataSource: "fileDropZoneMedia",
    autoFetchData: true,
    fields: [
        {name: "title", width: 120},
        // Show image thumbnail using viewFile URL
        {name: "image", title: "Preview", width: 80, type: "image",
         formatCellValue: function(value, record) {
             if (!record.image_filename) return "";
             var ds = isc.DataSource.get("fileDropZoneMedia");
             var imgURL = ds.getFileURL(record, "image");
             return "<img src='" + imgURL + "' width='60' height='45' style='object-fit:cover'/>";
         }
        },
        // Show filename with download link
        {name: "image_filename", title: "Filename", width: 120,
         formatCellValue: function(value, record) {
             if (!value) return "";
             var ds = isc.DataSource.get("fileDropZoneMedia");
             var downloadURL = ds.getFileURL(record, "image");
             return "<a href='" + downloadURL + "' target='_blank'>" + value + "</a>";
         }
        },
        {name: "image_filesize", title: "Size", width: 70,
         formatCellValue: function(value) {
             if (!value) return "";
             if (value < 1024) return value + " B";
             if (value < 1024 * 1024) return Math.round(value / 1024) + " KB";
             return (value / (1024 * 1024)).toFixed(1) + " MB";
         }
        }
    ],
    canRemoveRecords: true,
    warnOnRemoval: true,
    selectionChanged: function(record, state) {
        if (state) {
            uploadForm.editRecord(record);
        }
    }
});

isc.VLayout.create({
    width: "100%",
    membersMargin: 20,
    members: [
        isc.Label.create({
            contents: "<b>FileUploadItem Demo</b><br>" +
                      "Drag an image onto the drop zone or click to browse. " +
                      "The form uses <code>useFileUploadItem:true</code> to enable " +
                      "HTML5 drag-and-drop with thumbnail previews.<br>" +
                      "Select a record in the grid to edit it. The current image is shown separately.",
            height: 60,
            padding: 5
        }),
        isc.HLayout.create({
            membersMargin: 20,
            members: [
                isc.VLayout.create({
                    membersMargin: 10,
                    members: [uploadForm, currentImagePanel]
                }),
                mediaGrid
            ]
        })
    ]
});
