// FileDropZone showcase - demonstrates HTML5 file drag-and-drop with DataSource integration
// This sample shows a file management interface where files can be dropped, previewed,
// and saved to a DataSource using SmartClient databinding APIs.

// File details grid showing dropped files
isc.ListGrid.create({
    ID: "fileListGrid",
    width: 380,
    height: 150,
    showHeader: true,
    canReorderRecords: true,
    emptyMessage: "Drop files above to add them",
    selectionType: "single",
    fields: [
        {name: "icon", title: " ", width: 40, type: "icon",
         formatCellValue: function(value, record) {
             if (record.isImage) {
                 return "<img src='" + record.preview + "' width='30' height='30' style='object-fit:cover;border-radius:3px'/>";
             }
             return isc.Canvas.imgHTML("[SKINIMG]FileBrowser/file.png", 24, 24);
         }
        },
        {name: "name", title: "File Name", width: "*"},
        {name: "sizeStr", title: "Size", width: 80},
        {name: "type", title: "Type", width: 100}
    ],
    recordClick: function(viewer, record) {
        if (record.isImage && record.preview) {
            previewImg.setSrc(record.preview);
            previewLabel.setContents("<b>" + record.name + "</b><br>" + record.sizeStr);
        }
    }
});

// Preview panel for selected images
isc.Img.create({
    ID: "previewImg",
    width: 180,
    height: 135,
    imageType: "stretch",
    backgroundColor: "#f5f5f5",
    showEdges: true,
    src: "[SKINIMG]blank.gif"
});

isc.Label.create({
    ID: "previewLabel",
    width: 180,
    height: 40,
    align: "center",
    contents: "<i>Select an image to preview</i>"
});

isc.VLayout.create({
    ID: "previewPanel",
    width: 200,
    members: [
        isc.Label.create({contents: "<b>Preview</b>", height: 25}),
        previewImg,
        previewLabel
    ],
    membersMargin: 5
});

// Main drop zone
isc.FileDropZone.create({
    ID: "simpleDropZone",
    width: 380,
    height: 120,
    multiple: true,
    maxFiles: 10,
    maxFileSize: 10 * 1024 * 1024,  // 10MB per file
    showFileThumbnails: true,
    showImagePreviews: true,
    acceptedFileTypes: ["image/*", ".pdf", ".txt", ".doc", ".docx"],
    emptyDropAreaMessage: "Drop files here (images, PDFs, text, docs)",
    clickToAddMessage: "or click to browse",
    filesAdded: function(files) {
        updateFileList();
    }
});

// Update grid when files change
function updateFileList() {
    var files = simpleDropZone.getFiles();
    var records = [];

    for (var i = 0; i < files.length; i++) {
        var file = files[i];
        var isImage = file.type && file.type.indexOf("image/") === 0;

        var record = {
            name: file.name,
            size: file.size,
            sizeStr: formatSize(file.size),
            type: file.type || "Unknown",
            isImage: isImage,
            file: file
        };

        // Generate preview for images using blob URLs
        if (isImage && window.URL) {
            record.preview = URL.createObjectURL(file);
        }

        records.push(record);
    }

    fileListGrid.setData(records);
    updateSummary();
}

function formatSize(bytes) {
    if (bytes < 1024) return bytes + " B";
    if (bytes < 1024 * 1024) return (bytes / 1024).toFixed(1) + " KB";
    return (bytes / (1024 * 1024)).toFixed(1) + " MB";
}

function updateSummary() {
    var files = simpleDropZone.getFiles();
    var totalSize = simpleDropZone.getSize();
    summaryLabel.setContents(files.length + " file(s), " + formatSize(totalSize) + " total");
}

// Form for entering file metadata before saving
isc.DynamicForm.create({
    ID: "metadataForm",
    width: 380,
    numCols: 4,
    colWidths: [80, 140, 60, 100],
    fields: [
        {name: "title", title: "Title", width: 140,
         hint: "Enter title for upload", showHintInField: true},
        {name: "saveButton", type: "button", title: "Save to Server", startRow: false,
         click: function() {
             saveFilesToServer();
         }
        }
    ]
});

// Function to save files to the fileDropZoneMedia DataSource
// Uses DataSource.addData() with xhrUpload:true to upload File objects directly
function saveFilesToServer() {
    var files = simpleDropZone.getFiles();
    if (files.length === 0) {
        isc.say("Please add files first");
        return;
    }

    var title = metadataForm.getValue("title");
    if (!title) {
        isc.say("Please enter a title");
        return;
    }

    // Start processing animation
    simpleDropZone.startProcessing();

    // Upload first file for demo - in a real app you might upload all files
    var file = files[0];

    // Create a record with the file and metadata
    var record = {
        title: title,
        image: file
    };

    // Use DataSource.addData with xhrUpload:true to upload files directly
    // This bypasses the need for a form with file input
    isc.DataSource.get("fileDropZoneMedia").addData(record, function(response, data) {
        simpleDropZone.endProcessing();

        if (response.status === 0) {
            isc.say("Upload successful!<br>File saved to fileDropZoneMedia DataSource.");
            // Refresh the saved files grid
            savedFilesGrid.invalidateCache();
            // Clear the drop zone for next upload
            simpleDropZone.clearFiles();
            fileListGrid.setData([]);
            metadataForm.clearValues();
            previewImg.setSrc("[SKINIMG]blank.gif");
            previewLabel.setContents("<i>Select an image to preview</i>");
            updateSummary();
        } else {
            isc.warn("Upload failed: " + (response.data || "Unknown error"));
        }
    }, {xhrUpload: true});
}

// Summary and action buttons
isc.Label.create({
    ID: "summaryLabel",
    height: 25,
    contents: "0 file(s), 0 B total"
});

isc.IButton.create({
    ID: "clearButton",
    title: "Clear All",
    icon: "[SKINIMG]actions/remove.png",
    click: function() {
        simpleDropZone.clearFiles();
        fileListGrid.setData([]);
        previewImg.setSrc("[SKINIMG]blank.gif");
        previewLabel.setContents("<i>Select an image to preview</i>");
        updateSummary();
    }
});

// Grid showing previously saved files from the DataSource
isc.ListGrid.create({
    ID: "savedFilesGrid",
    width: "100%",
    height: 120,
    dataSource: "fileDropZoneMedia",
    autoFetchData: true,
    emptyMessage: "No files saved yet",
    fields: [
        {name: "title", width: 150},
        // For binary/imageFile fields, show download link using the filename
        {name: "image_filename", title: "File", width: 150,
         formatCellValue: function(value, record) {
             if (!value) return "";
             // Use DataSource.getFileURL to get the download URL
             var ds = isc.DataSource.get("fileDropZoneMedia");
             var downloadURL = ds.getFileURL(record, "image");
             return "<a href='" + downloadURL + "' target='_blank'>" + value + "</a>";
         }
        },
        {name: "image_filesize", title: "Size", width: 80,
         formatCellValue: function(value) {
             if (!value) return "";
             return formatSize(value);
         }
        }
    ],
    canRemoveRecords: true,
    warnOnRemoval: true
});

// Main layout
isc.VLayout.create({
    width: "100%",
    height: "100%",
    membersMargin: 10,
    padding: 10,
    members: [
        isc.Label.create({
            contents: "<b>File Drop Zone with DataSource Integration</b><br>" +
                      "Drop files onto the zone, enter a title, and save to the fileDropZoneMedia DataSource.",
            height: 40
        }),
        isc.HLayout.create({
            membersMargin: 20,
            members: [
                isc.VLayout.create({
                    membersMargin: 10,
                    members: [
                        simpleDropZone,
                        fileListGrid,
                        isc.HLayout.create({
                            membersMargin: 10,
                            height: 30,
                            members: [summaryLabel, clearButton]
                        }),
                        metadataForm
                    ]
                }),
                previewPanel
            ]
        }),
        isc.Label.create({
            contents: "<b>Saved Files</b> (from fileDropZoneMedia DataSource)",
            height: 25
        }),
        savedFilesGrid
    ]
});
