# Converting SmartClient JavaScript Samples to TypeScript

This guide explains how to convert SmartClient `.js` inline examples to `.ts` TypeScript samples.

## Module Isolation

Each `.ts` file must be a module to avoid global namespace conflicts with other samples.

**Add at the end of every file:**
```typescript
export {};
```

This makes the file a module even if it doesn't export anything.

## Global ID Declarations

When SmartClient components are created with an `ID` property, they become global variables. TypeScript needs to know about these.

**For each component with an ID, add a declaration before the create() call:**
```typescript
declare var myGrid: ListGrid;
isc.ListGrid.create({
    ID: "myGrid",
    ...
});
```

**Or use `const` with type assertion for components created inline:**
```typescript
const myGrid = isc.ListGrid.create({
    ...
}) as ListGrid;
```

## String Methods to Arrow Functions

SmartClient's "string method" feature (where event handlers can be specified as strings) is not supported in TypeScript. Convert all string methods to arrow functions.

### Simple Cases

**Before (JavaScript):**
```javascript
isc.IButton.create({
    title: "Click Me",
    click: "doSomething()"
});
```

**After (TypeScript):**
```typescript
isc.IButton.create({
    title: "Click Me",
    click: () => doSomething()
});
```

### With Component References

When the string references a component by ID, ensure the ID is declared:

**Before:**
```javascript
isc.IButton.create({
    click: "myGrid.fetchData()"
});
```

**After:**
```typescript
declare var myGrid: ListGrid;
// ... myGrid creation ...

isc.IButton.create({
    click: () => myGrid.fetchData()
});
```

### With `this` Context

Arrow functions capture lexical `this`, which differs from string methods where `this` refers to the component. Use the component reference directly:

**Before:**
```javascript
isc.IButton.create({
    ID: "myButton",
    click: "this.setTitle('Clicked')"
});
```

**After:**
```typescript
declare var myButton: IButton;
isc.IButton.create({
    ID: "myButton",
    click: () => myButton.setTitle('Clicked')
});
```

### With Event Parameters

Some handlers receive parameters. Check the API docs for the handler signature:

**Before:**
```javascript
isc.ListGrid.create({
    recordClick: "alert('Clicked: ' + record.name)"
});
```

**After:**
```typescript
isc.ListGrid.create({
    recordClick: (viewer, record) => alert('Clicked: ' + record.name)
});
```

## FormItem Type Property

The `type` property on form items accepts both data types (`"text"`, `"boolean"`) and editor class names (`"ButtonItem"`, `"CheckboxItem"`). For TypeScript, prefer using lowercase shortcuts which are included in the type:

```typescript
// These work:
{ name: "field1", type: "text" }
{ name: "field2", type: "boolean" }
{ name: "field3", type: "select" }
{ name: "field4", type: "button" }

// For explicit editor types, use editorType instead:
{ name: "field5", editorType: "SpinnerItem" }
```

## Array Properties with String IDs

Properties like `members`, `children`, `headerControls`, and `gridComponents` accept both component instances and string IDs. Both are valid in TypeScript:

```typescript
isc.HLayout.create({
    members: [
        myButton,           // Component reference
        "existingComponent" // String ID (must exist)
    ]
});
```

## Anonymous Classes / Custom Instance Methods

SmartClient instances behave like Java "anonymous classes": you can add custom properties,
add custom methods, and even call `Super()` from custom methods, all without formally
defining a new class via `defineClass()`.

To use this idiomatic pattern and retain full TypeScript type checking, declare an
interface that extends the parent class and includes your custom methods, then pass
the configuration as the second (unchecked) parameter to `create()` and use a type
assertion on the result.

### Basic Pattern

```typescript
// 1. Declare the extended interface with custom methods
interface SalesDataGrid extends isc.ListGrid {
    lastChart: isc.FacetChart;
    updateChart(chartType?: string): void;
}

// 2. Pass config as second (unchecked) param, cast result to extended interface
const salesDataGrid = isc.ListGrid.create(null, {
    updateChart: function(chartType) {
        if (this.lastChart) this.lastChart.destroy();
        this.lastChart = this.chartData("product");
    }
}) as SalesDataGrid;

// 3. Now TypeScript knows about the custom methods
salesDataGrid.updateChart();  // OK - no error
```

### Why the Second Parameter?

The first parameter to `create()` is type-checked. By passing `null` as the first
parameter and our custom config as the second, we bypass TypeScript's checking for
the custom properties while still getting type safety on the result via the interface.

### Complete Example

See the [gridChart sample](./charts/gridChart.ts) for a complete working example
that demonstrates:
- Declaring a custom interface
- Using `create(null, {...})` pattern
- Type-safe access to custom methods
- Calling `Super()` from custom methods

### When to Use

Use this pattern when:
- You need custom methods on a component instance
- The methods call `Super()` or interact with other instance properties
- You want TypeScript to provide autocomplete and type checking for those methods

### Quick Alternative: `as any`

For simpler cases where you just need to call an undocumented method once:
```typescript
(myGrid as any).someInternalMethod();
```

This is faster but provides no type safety. Prefer the Interface pattern for
any custom methods that are called multiple times or are central to the sample's logic.

## Type Assertions

Use type assertions when TypeScript can't infer the correct type:

### DataSource References

When a DataSource is referenced by ID string:
```typescript
isc.ListGrid.create({
    dataSource: "supplyItem" as unknown as DataSource
});
```

Or declare the DataSource:
```typescript
declare var supplyItem: DataSource;
isc.ListGrid.create({
    dataSource: supplyItem
});
```

### Dynamic Property Access

When accessing properties dynamically:
```typescript
const value = (record as any)[fieldName];
```

### Component Method Results

When methods return a generic type but you know the specific type:
```typescript
const form = myTabSet.getTab(0).pane as DynamicForm;
```

## External Data Variables

Many samples use external data that is loaded via separate `<script>` tags in the HTML files
that host the examples. These include:

- **Data arrays**: `countryData`, `eventData`, `chartData`, `employeeData`, etc.
- **DataSource instances**: `supplyItem`, `employees`, `worldDS`, etc.
- **Components created by XML layouts**: `pageLayout`, `categoryTree`, etc.

### Centralized Declarations File

Rather than adding `declare var` statements to each sample file, we use a centralized
`externalData.d.ts` file that declares all common external variables as `any`. This file
is automatically included by the TypeScript configuration.

**Location:** `inlineExamples/externalData.d.ts`

This approach:
- Avoids duplicating declarations across sample files
- Provides a single place to add new external variables
- Works automatically for all samples via tsconfig.json

### Adding New External Variables

If you create a new sample that uses an external data file not already declared,
add it to `externalData.d.ts`:

```typescript
// In externalData.d.ts
declare var myNewDataSource: any;
declare var myCustomData: any;
```

### Per-File Declarations (Alternative)

For sample-specific variables that shouldn't be in the shared file, you can still
add declarations directly in the sample:

```typescript
declare var countryData: any[];
declare var supplyItem: DataSource;
```

## Common Type Narrowing Patterns

### Record Properties

When accessing properties on records:
```typescript
// If the property might not exist:
if ('customProp' in record) {
    const val = (record as any).customProp;
}
```

### Callback Parameters

Handler callbacks often have `any` typed parameters. You can narrow them:
```typescript
selectionChanged: (record: ListGridRecord, state: boolean) => {
    // record is now properly typed
}
```

## Checklist for Conversion

1. [ ] Add `export {};` at end of file
2. [ ] Declare all component IDs used (`declare var name: Type;`)
3. [ ] Convert all string methods to arrow functions
4. [ ] Add any new external data variables to `externalData.d.ts`
5. [ ] Add type assertions where needed for DataSource references
6. [ ] Verify `this` references are converted to use component IDs
7. [ ] Run `tsc --noEmit` to check for remaining errors
8. [ ] Test the sample in the browser to verify runtime behavior

## Custom Properties on Component Instances

Some samples add custom properties or methods to component instances that aren't part of the
standard SmartClient API. TypeScript will flag these as errors (TS2353: "Object literal may
only specify known properties").

**Important:** Only use these approaches for truly sample-specific custom properties. If a
property actually exists in the framework but isn't documented, leave the compile warning
as-is so it can be addressed by documenting the property.

### For 1-2 Custom Properties: Use @ts-expect-error

When a component has just one or two custom properties, use `@ts-expect-error` with a comment
explaining why:

```typescript
isc.IButton.create({
    title: "Save",
    click: function() { this.saveData(); },
    // @ts-expect-error Sample-specific custom method for saving data
    saveData: function() { ... }
});
```

For nested custom properties:
```typescript
isc.DataSource.create({
    ID: "myDS",
    // @ts-expect-error width is a UI hint, not in DataSourceField type
    fields: [
        { name: "field1", width: 100 }
    ]
});
```

### For 3+ Custom Properties: Use Second Parameter to create()

When a component has three or more custom properties, move them to the second (untyped)
parameter of `create()` with a comment explaining the pattern:

```typescript
isc.IButton.create({
    title: "Save",
    click: function() { this.saveEditPane(); }
// Sample-specific custom methods - placed in second (untyped) parameter
}, {
    saveEditPane: function() { ... },
    restoreEditPane: function() { ... },
    validateBeforeSave: function() { ... }
});
```

**Note:** Documented framework methods being overridden (like `initWidget`, `draw`, etc.)
should remain in the first parameter. Only truly custom sample-specific methods go in the
second parameter.

### For Complex Cases: Create an Interface

When a component has many custom properties AND you need type checking on those properties:

```typescript
// Define interface with custom properties
interface PortletListGrid extends isc.ListGrid {
    gViewIconHTML: string;
    animatePortletTime: number;
    closePortlet: (portlet: any, record: any) => void;
}

// Use type parameter on create()
const portletList = isc.ListGrid.create<PortletListGrid>({
    ID: "portletList",
    gViewIconHTML: isc.Canvas.imgHTML("[SKIN]actions/view.png"),
    animatePortletTime: 750,
    closePortlet: function(portlet, record) { ... }
});

// Now TypeScript knows about custom properties
portletList.gViewIconHTML;  // OK
```

### When Using Custom Classes via defineClass()

If the sample defines a custom class with `defineClass()`, declare the class in the
global namespace and use type assertions:

```typescript
// Define the class
isc.defineClass("SimplePortlet", "Window").addMethods({ ... });

// Declare it for TypeScript
declare global { namespace isc { const SimplePortlet: typeof isc.Window; } }

// Use with custom properties via interface
interface SimplePortletInstance extends isc.Window {
    portletRecord: any;
    portletList: any;
}

var portlet = isc.SimplePortlet.create<SimplePortletInstance>({
    portletRecord: record,
    portletList: this
});
```

### DataSource Field UI Hints

Properties like `width` on DataSourceField are UI hints that aren't part of the
DataSourceField type. Either:

1. Move the width to the ListGrid's fields definition (preferred):
```typescript
// Instead of width in DataSource:
isc.DataSource.create({
    fields: [{ name: "status", type: "boolean" }]  // no width here
});
isc.ListGrid.create({
    dataSource: "myDS",
    fields: [{ name: "status", width: 40 }]  // width here
});
```

2. Or use `as any` on the fields array if width must stay in DataSource:
```typescript
isc.DataSource.create({
    fields: [
        { name: "status", type: "boolean", width: 40 }
    ] as any
});
```

## Runtime vs Documented Inheritance Edge Case

In rare cases, a property that works at runtime may not be recognized by TypeScript because the
class's documented inheritance differs from its actual runtime inheritance.

**Example: `titleStyle` on `IButton`**

The `titleStyle` property is documented on `StretchImgButton` with `@visibility external`, so it
should be available. However, `IButton`'s JSDoc declares `@inheritsFrom Button`, not
`StretchImgButton`. At runtime, depending on the skin, `IButton` may actually inherit from
`StretchImgButton`, making `titleStyle` valid.

Since TypeScript definitions follow the documented inheritance chain (`IButton` → `Button` →
`StatefulCanvas`), and `titleStyle` isn't in that chain, TypeScript reports an error even though
the property works at runtime.

**Solution:** Use `@ts-ignore` with an explanatory comment:

```typescript
isc.IButton.create({
    ID: "stretchButton",
    title: "Stretch Button",
    // @ts-ignore titleStyle is on StretchImgButton which IButton may extend at runtime
    titleStyle: "stretchTitle"
});
```

This edge case is rare - it only occurs when:
1. A property is documented on a class that isn't in the declared inheritance chain
2. The runtime inheritance differs from the documented inheritance
3. The difference is intentional (skin-dependent behavior)

## Troubleshooting Common Errors

### "Property 'X' does not exist on type 'Y'" (TS2353)
- Custom property: use `as any` for 1-2 properties, or create interface for 3+
- Undocumented property: use `(obj as any).X`
- UI hint on DataSourceField: move to ListGrid/FormItem or use `as any`

### "Property 'X' does not exist on type 'Y'"
- The property may be internal/undocumented - use `(obj as any).X`
- The property may be on a subclass - use proper type assertion

### "Type 'string' is not assignable to type '() => ...'"
- Convert string method to arrow function

### "Cannot find name 'X'"
- Add `declare var X: Type;` for global IDs or external variables

### "Type 'X' is not assignable to type 'Y'"
- Check if a type assertion is needed: `value as Type`
- For DataSources referenced by string: `"dsName" as unknown as DataSource`
