// Set the Reify server URL at the class level
// so all Reify operations in this session use
// the builder.  This respects any URL already
// set by the environment or hosting page, and
// avoids hardcoding a URL in per-call settings.
if (isc.Reify.serverURL == "https://reify.com")
{
    isc.Reify.setServerURL(
        "https://create.reify.com");
}

// Reify public sample credentials, shared
// with the other Reify Showcase examples.
// The server hosts read-only starter apps
// whose screen components have event-driven
// workflows that the editor can display.
var reifySettings = {
    userName: "reifySample",
    password: "tryReify",
    willHandleError: true,
    verifyAsError: true,
    verifyDataSources: true
};

// Reify sample apps whose screens contain
// event-driven workflows that demonstrate
// different workflow patterns.  Stored values
// are the project names sent to
// Reify.loadProject(); display values are the
// labels shown in the picker.
var reifyApps = {
    "(none)": "(none)",
    "Order Management Starter App":
        "Order Management",
    "Issue Tracking App (Simple)":
        "Issue Tracking",
    "Supply Catalog":
        "Supply Catalog",
    "Dashboard Starter App":
        "Dashboard"
};

// The WorkflowEditor provides a visual,
// interactive way to build and edit
// SmartClient Process workflows.
isc.WorkflowEditor.create({
    ID: "workflowEditor",
    width: "100%",
    height: "100%",
    backgroundColor: "#fafafa",
    elementBackgroundColor: "white",
    // Auto-refresh source tabs when the user
    // edits tasks (add, remove, edit) so XML
    // and JS views always reflect the current
    // process state.
    changed : function () {
        refreshSourceTabs();
    }
});

// Container for the loaded Reify app screen.
// The screen fills this VLayout automatically
// via the Layout sizing system.
isc.VLayout.create({
    ID: "screenContainer",
    width: "100%",
    height: "100%"
});

// ---- Source Tabs (XML and JS) ----
// Editable text areas showing the current
// workflow as Component XML or as a
// isc.Process.create() JS call.  Each tab
// has a Reload button so the user can edit
// the text and push changes back.

isc.DynamicForm.create({
    ID: "xmlArea",
    width: "100%",
    height: "100%",
    numCols: 1,
    fields: [{
        name: "source",
        editorType: "TextAreaItem",
        width: "*",
        height: "*",
        showTitle: false
    }]
});

isc.DynamicForm.create({
    ID: "jsArea",
    width: "100%",
    height: "100%",
    numCols: 1,
    fields: [{
        name: "source",
        editorType: "TextAreaItem",
        width: "*",
        height: "*",
        showTitle: false
    }]
});

// XML reload is async (server round-trip to
// convert XML to JS); JS reload is sync
// (client-side eval).
isc.IButton.create({
    ID: "xmlReloadButton",
    title: "Reload Editor from XML",
    autoFit: true,
    click : function () {
        var xml =
            xmlArea.getValue("source");
        if (!xml) return;
        statusLabel.setContents(
            "Loading XML...");
        workflowEditor.setProcessXML(xml,
            function (process) {
                if (!process) {
                    statusLabel.setContents(
                        "XML load failed "
                        + "&ndash; check "
                        + "syntax");
                    return;
                }
                statusLabel.setContents(
                    "Loaded from XML");
                refreshSourceTabs();
            }
        );
    }
});

isc.IButton.create({
    ID: "jsReloadButton",
    title: "Reload Editor from JS",
    autoFit: true,
    click : function () {
        var js =
            jsArea.getValue("source");
        if (!js) return;
        try {
            workflowEditor.setProcessJS(js);
            statusLabel.setContents(
                "Loaded from JS");
        } catch (e) {
            statusLabel.setContents(
                "JS error: " + e.message);
        }
        refreshSourceTabs();
    }
});

isc.Label.create({
    ID: "statusLabel",
    width: "100%",
    height: 22,
    contents: "Loading app...",
    padding: 4
});

isc.VLayout.create({
    ID: "xmlPane",
    width: "100%",
    height: "100%",
    members: [xmlReloadButton, xmlArea]
});

isc.VLayout.create({
    ID: "jsPane",
    width: "100%",
    height: "100%",
    members: [jsReloadButton, jsArea]
});

isc.TabSet.create({
    ID: "sourceTabs",
    width: "100%",
    height: "100%",
    tabs: [
        { title: "XML Source",
            pane: xmlPane },
        { title: "JS Source",
            pane: jsPane }
    ],
    tabSelected : function () {
        refreshSourceTabs();
    }
});

// ---- Toolbar ----
// UI Context picker selects which Reify app
// to load; Workflow picker shows the screen's
// event-driven workflows for that app.
isc.DynamicForm.create({
    ID: "workflowToolbar",
    width: "100%",
    numCols: 4,
    colWidths: [110, 250, 90, 350],
    items: [{
        name: "app",
        title: "UI Context",
        type: "select",
        width: 250,
        valueMap: reifyApps,
        defaultValue: "(none)",
        changed :
            function (form, item, value)
        {
            reifyAppSelected(value);
        }
    }, {
        name: "workflow",
        title: "Workflow",
        type: "select",
        width: 350,
        changed :
            function (form, item, value)
        {
            workflowSelected(value);
        }
    }]
});

// ---- Left Pane: Editor + Screen tabs ----
// The Editor tab shows the WorkflowEditor;
// the Screen tab shows the loaded Reify app
// so the user can see the UI whose workflows
// are being edited.
isc.TabSet.create({
    ID: "editorTabs",
    width: "100%",
    height: "100%",
    tabs: [
        { title: "Editor",
            pane: workflowEditor },
        { title: "Screen",
            pane: screenContainer }
    ]
});

// ---- Main Layout ----
// Toolbar on top; below that, the editor/
// screen tabs (70%) and source tabs (30%)
// side by side with a resize bar.
isc.VLayout.create({
    ID: "workflowLayout",
    width: "100%",
    height: "100%",
    members: [
        workflowToolbar,
        isc.HLayout.create({
            width: "100%",
            height: "100%",
            members: [
                isc.VLayout.create({
                    width: "70%",
                    height: "100%",
                    showResizeBar: true,
                    members: [editorTabs]
                }),
                isc.VLayout.create({
                    width: "30%",
                    height: "100%",
                    members: [
                        sourceTabs,
                        statusLabel
                    ]
                })
            ]
        })
    ]
});

// ---- Helpers ----

// Refresh XML and JS source text areas from
// the editor's current process state.
// Serialization can fail on partially-built
// workflows so each path uses try/catch.
function refreshSourceTabs() {
    if (!workflowEditor.process) return;

    try {
        var xml =
            workflowEditor.getProcessXML();
        xmlArea.setValue(
            "source", xml || "");
    } catch (e) {
        xmlArea.setValue("source",
            "// Error getting XML: " + e);
    }

    try {
        var props =
            workflowEditor.getProcessJS();
        if (props) {
            jsArea.setValue("source",
                "isc.Process.create(" +
                isc.JSON.encode(props, {
                    prettyPrint: true
                }) + ")");
        } else {
            jsArea.setValue("source", "");
        }
    } catch (e) {
        jsArea.setValue("source",
            "// Error getting JS: " + e);
    }
}

// Handle UI Context picker selection.
// Loads the chosen Reify app into the Screen
// tab and rebuilds the Workflow picker with
// that app's event-driven workflows.
function reifyAppSelected(value) {
    if (value == "(none)") {
        clearScreen();
        workflowEditor.setRuleScope(null);
        workflowEditor.setProperties({
            dataSources: null,
            availableComponents: null
        });
        updateWorkflowPicker(null);
        statusLabel.setContents(
            "Context cleared");
        editorTabs.selectTab(0);
        return;
    }
    loadReifyApp(value);
}

// Handle workflow picker selection.  Screen
// event keys are encoded as
// "screenEvent:componentID:eventName".
function workflowSelected(value) {
    if (value.indexOf("screenEvent:") == 0) {
        loadScreenEvent(value);
        return;
    }
}

// Load a screen event's workflow into the
// editor.  Decodes the component ID and event
// name from the picker key, finds the matching
// event via getEvents(), then loads the Process
// definition (or converts an Action).
function loadScreenEvent(key) {
    var parts = key.split(":");
    if (parts.length < 3) return;
    var componentId = parts[1],
        eventName = parts[2];

    var screen = window._currentScreen;
    if (!screen) return;

    // Find the matching event definition
    var events = screen.getEvents();
    var match = null;
    for (var i = 0; i < events.length; i++) {
        if (events[i].source &&
            events[i].source.ID ==
                componentId &&
            events[i].eventName == eventName)
        {
            match = events[i];
            break;
        }
    }
    if (!match || !match.event) {
        statusLabel.setContents(
            "Event not found");
        return;
    }

    // Load the event definition into the
    // editor: Process configs go directly;
    // Actions are converted via the editor's
    // convertActionToProcess utility
    var eventDef = match.event;
    var label =
        match.source.getClassName() +
        "." + eventName;
    if (eventDef._constructor == "Process") {
        workflowEditor.setProcess(
            isc.Process.create(eventDef));
        statusLabel.setContents(
            "Editing: " + label);
    } else {
        var proc =
            workflowEditor
                .convertActionToProcess(
                    eventDef);
        if (proc) {
            workflowEditor.setProcess(proc);
            statusLabel.setContents(
                "Editing: " + label);
        } else {
            workflowEditor.clearProcess();
            statusLabel.setContents(
                "Could not convert: " +
                label);
        }
    }
    refreshSourceTabs();
    // Switch to Editor tab to show workflow
    editorTabs.selectTab(0);
}

// Populate the workflow picker with events
// discovered on the loaded Reify screen.
// Each component event handler containing a
// Process or Action becomes a selectable entry.
function updateWorkflowPicker(screen) {
    var vm = {};

    if (screen) {
        var events = screen.getEvents();
        for (var i = 0;
            i < events.length; i++)
        {
            var evt = events[i];
            if (!evt.source ||
                !evt.source.ID)
            {
                continue;
            }
            var key = "screenEvent:" +
                evt.source.ID + ":" +
                evt.eventName;
            var label =
                evt.source.getClassName() +
                "." + evt.eventName;
            if (evt.event &&
                evt.event._constructor ==
                    "Process")
            {
                label += " [Workflow]";
            } else {
                label += " [Action]";
            }
            vm[key] = label;
        }
    }

    workflowToolbar.getItem("workflow")
        .setValueMap(vm);
}

// Recursively gather screen components that
// workflow tasks can target (forms, grids,
// buttons).  Excludes internal sub-components
// like scrollbars, grid bodies, and filter
// editors by walking Layout members rather
// than all children.
function gatherScreenComponents(
    canvas, list)
{
    if (!canvas) return;
    if (canvas.ID &&
        !isc.isA.Scrollbar(canvas) &&
        !isc.isA.Splitbar(canvas) &&
        (isc.isA.DynamicForm(canvas) ||
            isc.isA.ListGrid(canvas) ||
            !isc.isA.Layout(canvas)))
    {
        var id = canvas.ID;
        // Skip internal sub-components
        if (id.indexOf("_body") < 0 &&
            id.indexOf("_filterEditor") < 0 &&
            id.indexOf("_sorter") < 0 &&
            id.indexOf("_hscroll") < 0 &&
            id.indexOf("_vscroll") < 0 &&
            id.indexOf("_summaryRow") < 0)
        {
            list.add({
                ID: id,
                className:
                    canvas.getClassName(),
                description: id + " (" +
                    canvas.getClassName() +
                    ")"
            });
        }
    }
    // Walk members (explicit children) to
    // avoid internal sub-components; fall
    // back to children for non-Layouts
    var kids = canvas.members ||
        canvas.children || [];
    for (var i = 0; i < kids.length; i++) {
        gatherScreenComponents(
            kids[i], list);
    }
}

// Remove all members from the screen
// container, destroying the previous Reify
// screen so a new one can be loaded cleanly
function clearScreen() {
    var members =
        screenContainer.getMembers();
    if (members && members.length > 0) {
        var toDestroy = members.duplicate();
        for (var i = 0;
            i < toDestroy.length; i++)
        {
            screenContainer.removeMember(
                toDestroy[i]);
            toDestroy[i].destroy();
        }
    }
    window._currentScreen = null;
}

// Load a Reify project, create its first
// screen in the Screen tab, and provide
// DataSources and components to the editor
// so that Widget, Form, and Grid tasks become
// available in the task picker.  Then populate
// the Workflow picker with the screen's
// event-driven workflows and auto-select the
// first one.
function loadReifyApp(projectName) {
    statusLabel.setContents(
        "Loading app...");

    isc.Reify.loadProject(projectName,
        function (project, projects,
            rpcResponse)
        {
            var message =
                isc.RPCManager
                    .getLoadProjectErrorMessage(
                        rpcResponse);
            if (message) {
                statusLabel.setContents(
                    "Error: " + message);
                return;
            }

            // Clear any previously loaded
            // screen before creating new one
            clearScreen();

            // Create the first screen and add
            // it to the Screen tab container
            var screenName =
                project.screens[0].ID;
            var screen =
                project.createScreen(
                    screenName);
            screen.setWidth("100%");
            screen.setHeight("100%");
            screenContainer.addMember(screen);
            window._currentScreen = screen;

            // Gather DataSources for the
            // editor
            var dsList =
                isc.DS.getAppDataSources();
            var dsNames = [];
            for (var i = 0;
                i < dsList.length; i++)
            {
                dsNames.add(dsList[i].ID);
            }

            // Gather screen components that
            // workflow tasks can target
            var components = [];
            gatherScreenComponents(
                screen, components);

            // Provide context to the editor so
            // that task types referencing forms,
            // grids, and DataSources become
            // available in the task picker
            workflowEditor.setProperties({
                dataSources: dsNames,
                availableComponents: components
            });
            workflowEditor.setRuleScope(
                screen.ID);

            // Populate the workflow picker with
            // event-driven workflows from the
            // screen's component event handlers
            updateWorkflowPicker(screen);

            // Switch to Screen tab so the user
            // sees the loaded app
            editorTabs.selectTab(1);
            statusLabel.setContents(
                "Loaded: " +
                dsNames.length + " DS, " +
                components.length +
                " components");

            // Auto-load the first available
            // workflow into the editor
            var vm = workflowToolbar
                .getItem("workflow")
                .getValueMap();
            var keys = isc.getKeys(vm);
            for (var j = 0;
                j < keys.length; j++)
            {
                if (keys[j].indexOf(
                    "screenEvent:") == 0)
                {
                    workflowToolbar.setValue(
                        "workflow", keys[j]);
                    workflowSelected(keys[j]);
                    break;
                }
            }
        },
        reifySettings
    );
}

// ---- Startup ----
// Auto-load the first Reify app so the Screen
// tab has content and Widget/Form/Grid tasks
// are available immediately.  The UI Context
// picker is set to match.
var firstApp = isc.getKeys(reifyApps);
for (var _i = 0; _i < firstApp.length; _i++)
{
    if (firstApp[_i] != "(none)") {
        workflowToolbar.setValue(
            "app", firstApp[_i]);
        loadReifyApp(firstApp[_i]);
        break;
    }
}
