// ---- Custom Alert Task ----
// A ProcessElement subclass representing a
// configurable alert dialog.  Demonstrates how
// to create a custom workflow task with its own
// editor.  See +link{group:customTaskEditors}.

isc.defineClass(
    "CustomAlertTask", "ProcessElement"
).addProperties({
    classDescription: "Show a configurable " +
        "alert dialog",
    editorType: "CustomAlertTaskEditor",

    alertTitle: "Alert",
    iconStyle: "info",
    waitForDismissal: true
});

isc.CustomAlertTask.addMethods({
    // Label shown on the workflow diagram element
    getElementDescription : function () {
        return this.alertMessage ||
            "Show alert";
    },

    // Show the alert dialog.  Returns true for
    // synchronous (fire-and-forget) or null for
    // async (workflow waits for user dismissal).
    executeElement : function (process) {
        var title = this.alertTitle || "Alert";
        var msg = this.alertMessage || "";
        if (this.waitForDismissal) {
            isc.say(msg, function () {
                process.start();
            }, { title: title });
            return null;
        }
        isc.say(msg, null, { title: title });
        return true;
    }
});

// ---- Custom Alert Task Editor ----
// A WorkflowTaskEditor subclass providing form
// fields for the custom task properties.
// Override four methods to integrate with the
// WorkflowEditor lifecycle:
//   getEditorComponents()
//   setEditorValuesFromDefaults()
//   getEditorValuesAsDefaults()
//   validate()

isc.defineClass(
    "CustomAlertTaskEditor", "WorkflowTaskEditor"
).addProperties({

    // Properties this editor modifies; any
    // other properties on the task are preserved
    // automatically by the base class
    editProperties: [
        "ID", "description",
        "alertTitle", "alertMessage",
        "iconStyle", "waitForDismissal"
    ],

    // Build the editor form.  createIDEditor(),
    // createDescriptionEditor(), and
    // createButtonsComponent() are inherited
    // convenience methods from WorkflowTaskEditor.
    getEditorComponents : function () {
        this.alertForm = isc.DynamicForm.create({
            numCols: 2,
            colWidths: [120, "*"],
            fields: [{
                name: "alertTitle",
                title: "Alert Title",
                type: "text",
                required: true,
                defaultValue: "Alert",
                width: "*"
            }, {
                name: "alertMessage",
                title: "Message",
                type: "textArea",
                required: true,
                width: "*",
                height: 60
            }, {
                name: "iconStyle",
                title: "Icon",
                type: "select",
                valueMap: {
                    "info": "Info",
                    "warn": "Warning",
                    "error": "Error"
                },
                defaultValue: "info"
            }, {
                name: "waitForDismissal",
                title: "Wait for OK",
                type: "boolean",
                defaultValue: true
            }]
        });
        return [
            this.createIDEditor(),
            this.createDescriptionEditor(),
            this.alertForm,
            this.createButtonsComponent()
        ];
    },

    // Populate the custom form from the task's
    // existing saved values when editing
    setEditorValuesFromDefaults :
        function (defaults)
    {
        this.Super(
            "setEditorValuesFromDefaults",
            arguments);
        if (!defaults) return;
        this.alertForm.setValues({
            alertTitle:
                defaults.alertTitle || "Alert",
            alertMessage:
                defaults.alertMessage || "",
            iconStyle:
                defaults.iconStyle || "info",
            waitForDismissal:
                defaults.waitForDismissal !== false
        });
    },

    // Merge custom form values into the defaults
    // object returned by the base class
    getEditorValuesAsDefaults : function () {
        var defaults = this.Super(
            "getEditorValuesAsDefaults",
            arguments);
        var vals = this.alertForm.getValues();
        defaults.alertTitle = vals.alertTitle;
        defaults.alertMessage = vals.alertMessage;
        defaults.iconStyle = vals.iconStyle;
        defaults.waitForDismissal =
            vals.waitForDismissal;
        return defaults;
    },

    // Validate both base class fields and the
    // custom alert form
    validate : function () {
        return this.Super("validate", arguments)
            && this.alertForm.validate();
    }
});

// ---- WorkflowEditor instance ----
// Creates the editor, then registers the custom
// task and its folder so it appears in the task
// picker alongside the built-in task types.

isc.WorkflowEditor.create({
    ID: "workflowEditor",
    width: "100%",
    height: "100%",
    backgroundColor: "#fafafa",
    elementBackgroundColor: "white",
    changed : function () {
        refreshSourceTabs();
    }
});

// Register a "Custom Tasks" folder and the
// Custom Alert task within it.  The taskType
// links the picker entry to the ProcessElement
// subclass; editorType on the class links the
// task to its editor.
workflowEditor.registerFolderDescriptor({
    ID: "customTasksFolder",
    title: "Custom Tasks",
    icon: "workflow/advancedTasks.png"
});
workflowEditor.registerTaskDescriptor({
    ID: "customAlertDesc",
    taskType: "CustomAlertTask",
    title: "Custom Alert",
    description:
        "Show a configurable alert dialog",
    icon: "workflow/showMessage.png",
    taskPath: "customTasksFolder"
});

// ---- Source Tab (JS) ----
// An editable text area showing the current
// workflow as an isc.Process.create() JS call.

isc.DynamicForm.create({
    ID: "jsArea",
    width: "100%",
    height: "100%",
    numCols: 1,
    fields: [{
        name: "source",
        editorType: "TextAreaItem",
        width: "*",
        height: "*",
        showTitle: false
    }]
});

isc.IButton.create({
    ID: "jsReloadButton",
    title: "Reload Editor from JS",
    autoFit: true,
    click : function () {
        var js =
            jsArea.getValue("source");
        if (!js) return;
        try {
            workflowEditor.setProcessJS(js);
            statusLabel.setContents(
                "Loaded from JS");
        } catch (e) {
            statusLabel.setContents(
                "JS error: " + e.message);
        }
        refreshSourceTabs();
    }
});

isc.Label.create({
    ID: "statusLabel",
    width: "100%",
    height: 22,
    contents: "Ready",
    padding: 4
});

// ---- Layout ----
// Editor on the left (70%), source + status on
// the right (30%) with a resize bar.

isc.VLayout.create({
    ID: "jsPane",
    width: "100%",
    height: "100%",
    members: [jsReloadButton, jsArea]
});

isc.VLayout.create({
    ID: "mainLayout",
    width: "100%",
    height: "100%",
    members: [
        isc.HLayout.create({
            width: "100%",
            height: "100%",
            members: [
                isc.VLayout.create({
                    width: "70%",
                    height: "100%",
                    showResizeBar: true,
                    members: [workflowEditor]
                }),
                isc.VLayout.create({
                    width: "30%",
                    height: "100%",
                    members: [
                        jsPane,
                        statusLabel
                    ]
                })
            ]
        })
    ]
});

// ---- Helpers ----

// Refresh the JS source text area from the
// editor's current process state
function refreshSourceTabs() {
    if (!workflowEditor.process) return;
    try {
        var props =
            workflowEditor.getProcessJS();
        if (props) {
            jsArea.setValue("source",
                "isc.Process.create(" +
                isc.JSON.encode(props, {
                    prettyPrint: true
                }) + ")");
        } else {
            jsArea.setValue("source", "");
        }
    } catch (e) {
        jsArea.setValue("source",
            "// Error getting JS: " + e);
    }
}

// ---- Starter Workflow ----
// Load a sample workflow that uses the custom
// task so the editor is not empty on startup.

var sampleWorkflow = isc.Process.create({
    startElement: "start",
    sequences: [{
        _constructor: "ProcessSequence",
        ID: "main",
        elements: [{
            _constructor: "ScriptTask",
            ID: "start",
            description: "Initialize data",
            execute : function (input, process)
            {
                return { status: "ok" };
            }
        }, {
            _constructor: "CustomAlertTask",
            ID: "showAlert",
            alertTitle: "Welcome",
            alertMessage:
                "This workflow uses a " +
                "custom task!",
            iconStyle: "info",
            waitForDismissal: true
        }, {
            _constructor: "CustomAlertTask",
            ID: "confirm",
            alertTitle: "Confirm",
            alertMessage:
                "Processing complete.",
            iconStyle: "warn",
            waitForDismissal: false
        }]
    }]
});

workflowEditor.setProcess(sampleWorkflow);
refreshSourceTabs();
