// Multi-File Upload with Master-Detail Pattern
// This sample demonstrates FileUploadItem with multiple:true and a linked detail DataSource.
// Each dropped file is uploaded as a separate record in the detail DataSource, linked to
// the master record via a foreign key.

// Grid showing master records
isc.ListGrid.create({
    ID: "masterGrid",
    width: 450,
    height: 180,
    dataSource: "multiFileUploadMaster",
    autoFetchData: true,
    canRemoveRecords: true,
    warnOnRemoval: true,
    warnOnRemovalMessage: "Delete this record and all its attached files?",
    fields: [
        {name: "id", width: 50},
        {name: "title", width: 200},
        {name: "description", width: "*"}
    ],
    selectionChanged: function(record, state) {
        if (state) {
            // Load detail files for selected master
            detailGrid.fetchData({masterId: record.id});
            uploadForm.editRecord(record);
        }
    }
});

// Grid showing detail files for selected master record
isc.ListGrid.create({
    ID: "detailGrid",
    width: 450,
    height: 150,
    dataSource: "multiFileUploadDetail",
    canRemoveRecords: true,
    emptyMessage: "Select a master record to see its files",
    fields: [
        {name: "file_filename", title: "Filename", width: 200,
         formatCellValue: function(value, record) {
             if (!value) return "";
             var ds = isc.DataSource.get("multiFileUploadDetail");
             var downloadURL = ds.getFileURL(record, "file");
             return "<a href='" + downloadURL + "' target='_blank'>" + value + "</a>";
         }
        },
        {name: "file_filesize", title: "Size", width: 100,
         formatCellValue: function(value) {
             if (!value) return "";
             if (value < 1024) return value + " B";
             if (value < 1024 * 1024) return Math.round(value / 1024) + " KB";
             return (value / (1024 * 1024)).toFixed(1) + " MB";
         }
        },
        {name: "file_date_created", title: "Uploaded", width: 130}
    ]
});

// Upload form with FileUploadItem configured for multi-file master-detail uploads
isc.DynamicForm.create({
    ID: "uploadForm",
    width: 400,
    dataSource: "multiFileUploadMaster",
    useFileUploadItem: true,
    fields: [
        {name: "title", title: "Title", required: true},
        {name: "description", title: "Description", type: "textArea", height: 50},
        {name: "files", title: "Files", editorType: "FileUploadItem",
         // Enable multi-file mode with master-detail pattern:
         // multiple:true tells FileUploadItem to store files in a related DataSource
         multiple: true,
         // The detail DataSource where files will be stored - must have a foreignKey
         // field pointing to this form's DataSource (multiFileUploadMaster)
         dataSource: "multiFileUploadDetail",
         // Allow up to 10 files per master record
         maxFiles: 10,
         height: 140,
         showFileThumbnails: true,
         showImagePreviews: true,
         emptyDropAreaMessage: "Drop files here",
         clickToAddMessage: "or click to browse"
        }
    ]
});

// Button bar for form actions
isc.HLayout.create({
    ID: "buttonBar",
    membersMargin: 10,
    height: 30,
    members: [
        isc.IButton.create({
            title: "New Record",
            click: function() {
                uploadForm.editNewRecord();
                detailGrid.setData([]);
            }
        }),
        isc.IButton.create({
            title: "Save",
            click: function() {
                if (uploadForm.validate()) {
                    uploadForm.saveData(function(response, data) {
                        if (response.status === 0) {
                            isc.say("Saved successfully!");
                            masterGrid.invalidateCache();
                            if (data && data.id) {
                                detailGrid.fetchData({masterId: data.id});
                            }
                        }
                    });
                }
            }
        })
    ]
});

// Main layout
isc.VLayout.create({
    width: "100%",
    height: "100%",
    membersMargin: 15,
    padding: 10,
    members: [
        isc.Label.create({
            contents: "<b>Multi-File Upload with Master-Detail Pattern</b><br>" +
                      "This demonstrates <code>FileUploadItem</code> with <code>multiple:true</code> " +
                      "and a linked detail DataSource.<br>" +
                      "Each file is stored as a separate record in <code>multiFileUploadDetail</code>, " +
                      "linked to the master via <code>foreignKeyField</code>.",
            height: 60,
            padding: 5
        }),
        isc.HLayout.create({
            membersMargin: 20,
            members: [
                isc.VLayout.create({
                    membersMargin: 10,
                    members: [
                        isc.Label.create({contents: "<b>Upload Form:</b>", height: 20}),
                        uploadForm,
                        buttonBar
                    ]
                }),
                isc.VLayout.create({
                    membersMargin: 10,
                    members: [
                        isc.Label.create({contents: "<b>Master Records:</b>", height: 20}),
                        masterGrid,
                        isc.Label.create({contents: "<b>Files for Selected Master:</b>", height: 20}),
                        detailGrid
                    ]
                })
            ]
        })
    ]
});

// Start with a new record ready for input
uploadForm.editNewRecord();
