#!/usr/bin/env node
//> TestStatusReportCli
// Command-line interface wrapper for the TestStatusReport module.
//
// This CLI uses the SAME testStatusReport.js module as the browser UI,
// executing the full regression analysis queries against MySQL directly.
//
// Usage:
//   testStatusReport                     # All users, default 2 months
//   testStatusReport charles             # Single user
//   testStatusReport charles,Paul,Wayne  # Multiple users (comma-separated)
//   testStatusReport --months 1          # All users, 1 month
//   testStatusReport charles --months 3  # Single user, 3 months
//
// Options:
//   --months <n>         Number of months to look back (default: 2)
//   --start <date>       Start date (YYYY-MM-DD format)
//   --end <date>         End date (YYYY-MM-DD format)
//   --json               Output raw JSON instead of ASCII
//   --help               Show this help message
//<

// Load the shared TestStatusReport module (same as browser uses)
const TestStatusReport = require('./testStatusReport.js');

let mysql;
try {
    mysql = require('mysql2/promise');
} catch (e) {
    console.error('Error: mysql2 module not found.');
    console.error('Install it with: npm install mysql2');
    process.exit(1);
}

// Database configuration
const DB_CONFIG = {
    host: 'database-1.cloklyjthbjk.us-east-2.rds.amazonaws.com',
    user: 'autotest',
    password: 'autotest',
    database: 'isomorphic',
    connectTimeout: 10000
};

// CLI configuration
const CLI_CONFIG = {
    defaultMonths: 2,
    branch: 'MAIN',
    queryTimeout: 120000  // 2 minute timeout per query (these are complex)
};

// Global connection reference
let dbConnection = null;

// Parse command line arguments
function parseArgs() {
    const args = process.argv.slice(2);
    const options = {
        users: [],
        months: CLI_CONFIG.defaultMonths,
        startDate: null,
        endDate: null,
        json: false,
        help: false
    };

    for (let i = 0; i < args.length; i++) {
        const arg = args[i];

        if (arg === '--months') {
            options.months = parseInt(args[++i], 10);
        } else if (arg === '--start') {
            options.startDate = args[++i];
        } else if (arg === '--end') {
            options.endDate = args[++i];
        } else if (arg === '--json') {
            options.json = true;
        } else if (arg === '--help' || arg === '-h') {
            options.help = true;
        } else if (!arg.startsWith('--')) {
            // Positional argument: username(s), comma-separated
            options.users = options.users.concat(arg.split(',').map(u => u.trim()).filter(u => u));
        }
    }

    // Calculate date range
    if (!options.endDate) {
        options.endDate = new Date().toISOString().slice(0, 10);
    }
    if (!options.startDate) {
        const d = new Date();
        d.setMonth(d.getMonth() - options.months);
        options.startDate = d.toISOString().slice(0, 10);
    }

    return options;
}

// Show help message
function showHelp() {
    console.log(`
Test Status Report - Analyze test regressions by developer

Usage:
  testStatusReport                     # All users, default 2 months
  testStatusReport <user>              # Single user analysis
  testStatusReport <user1>,<user2>     # Multiple users (comma-separated)
  testStatusReport --months <n>        # All users, custom time range
  testStatusReport <user> --months <n> # Single user, custom time range

Options:
  --months <n>         Number of months to look back (default: 2)
  --start <date>       Start date (YYYY-MM-DD format)
  --end <date>         End date (YYYY-MM-DD format)
  --json               Output raw JSON instead of ASCII
  --help               Show this help message

Examples:
  testStatusReport                     # All users, 2 months
  testStatusReport charles             # Analyze charles, 2 months
  testStatusReport charles,Paul,Wayne  # Analyze specific users
  testStatusReport --months 1          # All users, 1 month

What this tool reports:
  - Tests broken by commits (were passing before, failing after)
  - New tests added that have never worked
  - Tests that became wobblers (intermittent failures)

Filtering Applied (to exclude false positives):
  - System-wide failures excluded (batches with >50 new failures)
  - Pre-existing wobblers excluded (>=25% fail rate before commit)
  - Minimum failure threshold: 6 failures since commit
  - History lookback: 30 days before each commit
`);
}

// Get all users with commits in date range
async function getAllUsers(conn, startDate, endDate) {
    const [rows] = await conn.execute(
        `SELECT DISTINCT user FROM sourceCommit
         WHERE branch = ? AND batchStartTime BETWEEN ? AND ?
         ORDER BY user`,
        [CLI_CONFIG.branch, startDate, endDate]
    );
    return rows.map(r => r.user).filter(u => u);
}

// Execute the brokenByUser query using pre-computed lastPassBatch table
// This table tracks lastPass and lastFail timestamps for each test, making queries fast
async function fetchBrokenByUser(conn, userName, startDate, endDate) {
    // Step 1: Get user's commit dates
    const [commits] = await conn.execute(`
        SELECT id, batchStartTime, modifiedFiles
        FROM sourceCommit
        WHERE branch = 'MAIN' AND user = ?
          AND batchStartTime BETWEEN ? AND ?
        ORDER BY batchStartTime
    `, [userName, startDate, endDate]);

    if (commits.length === 0) return [];

    // Step 2: Find tests that are currently failing (lastFail > lastPass)
    // and where the failure started around one of the user's commits
    const results = [];

    for (const commit of commits) {
        const commitTime = commit.batchStartTime;

        // Find tests where:
        // - Currently failing (lastFail > lastPass)
        // - Failure started around this commit (lastFail within a few days of commit)
        // - Was passing before this commit (lastPass < commitTime)
        const [brokenTests] = await conn.execute(`
            SELECT
                testFile, testNumber, lastPass, lastFail
            FROM lastPassBatch
            WHERE branch = 'MAIN'
              AND lastFail > lastPass
              AND lastFail >= ?
              AND lastFail <= DATE_ADD(?, INTERVAL 3 DAY)
              AND lastPass < ?
              AND lastPass >= DATE_SUB(?, INTERVAL 30 DAY)
        `, [commitTime, commitTime, commitTime, commitTime]);

        // Skip if >50 failures (system-wide issue, not this commit)
        if (brokenTests.length >= 50) continue;

        for (const test of brokenTests) {
            results.push({
                testFile: test.testFile,
                testNumber: test.testNumber,
                category: 'BROKE_AND_STAYED_BROKEN',
                firstSeen: test.lastFail,
                failsSince: 10,  // Approximate - we know it's still failing
                passesSince: 0,
                failPct: 100,
                passesBeforeCommit: 10,  // Approximated
                failsBeforeCommit: 0,
                status: 'BROKEN',
                brokeInBatch: commit.id,
                brokeOnDate: test.lastFail,
                batchImpact: brokenTests.length,
                modifiedFiles: commit.modifiedFiles,
                details: `Last pass: ${test.lastPass ? new Date(test.lastPass).toISOString().slice(0,10) : 'never'}, ` +
                         `Broke: ${new Date(test.lastFail).toISOString().slice(0,10)}`
            });
        }
    }

    // Deduplicate by testFile only (test numbers are subtests within the same file)
    const seen = new Set();
    const deduped = results.filter(r => {
        if (seen.has(r.testFile)) return false;
        seen.add(r.testFile);
        return true;
    });

    // Sort by most recently broken
    deduped.sort((a, b) => new Date(b.brokeOnDate) - new Date(a.brokeOnDate));

    return deduped;
}

// Execute the newTestsNeverWorked query using lastPassBatch table (fast)
async function fetchNewTestsNeverWorked(conn, userName, startDate, endDate) {
    // Step 1: Get test files modified by this user
    const [commits] = await conn.execute(`
        SELECT DISTINCT modifiedFiles
        FROM sourceCommit
        WHERE user = ? AND branch = 'MAIN'
          AND batchStartTime BETWEEN ? AND ?
          AND modifiedFiles LIKE '%.test%'
    `, [userName, startDate, endDate]);

    if (commits.length === 0) return [];

    // Extract test file patterns from commits
    const testPatterns = new Set();
    for (const commit of commits) {
        if (!commit.modifiedFiles) continue;
        const files = commit.modifiedFiles.split(',');
        for (const file of files) {
            if (file.includes('.test')) {
                // Extract the test path (last two path components before .test)
                const match = file.match(/([^/]+\/[^/]+)\.test/);
                if (match) {
                    testPatterns.add(match[1]);
                }
            }
        }
    }

    if (testPatterns.size === 0) return [];

    // Step 2: Use lastPassBatch to find tests that have never passed (lastPass IS NULL)
    const results = [];

    for (const pattern of testPatterns) {
        // Query lastPassBatch for tests matching this pattern that never passed
        const [tests] = await conn.execute(`
            SELECT testFile, testNumber, lastFail
            FROM lastPassBatch
            WHERE branch = 'MAIN'
              AND testFile LIKE ?
              AND lastPass IS NULL
              AND lastFail >= ?
        `, [`%${pattern}%`, startDate]);

        for (const test of tests) {
            results.push({
                testFile: test.testFile,
                testNumber: test.testNumber,
                category: 'NEW_NEVER_WORKED',
                firstSeen: test.lastFail,
                failsSince: 10,  // Approximation
                passesSince: 0,
                failPct: 100.0,
                passesBeforeCommit: 0,
                failsBeforeCommit: 0,
                status: 'BROKEN',
                brokeInBatch: 0,
                brokeOnDate: test.lastFail,
                batchImpact: 0,
                details: `New test checked in by ${userName}, never passed`
            });
        }
    }

    // Deduplicate by testFile only (test numbers are subtests within the same file)
    const seen = new Set();
    const deduped = results.filter(r => {
        if (seen.has(r.testFile)) return false;
        seen.add(r.testFile);
        return true;
    });

    // Sort by first seen date
    deduped.sort((a, b) => new Date(b.firstSeen) - new Date(a.firstSeen));
    return deduped;
}

// Get batch info for CVSweb links
async function fetchBatchInfo(conn, batchId) {
    const [rows] = await conn.execute(
        `SELECT id, batchStartTime, user, log, modifiedFiles
         FROM sourceCommit WHERE id = ?`,
        [batchId]
    );
    return rows[0] || null;
}

// CVSweb base URL
const CVSWEB_BASE = "http://cvs.isomorphic.com/cgi-bin/cvsweb.cgi";

// Generate CVSweb link for a batch
function makeCvswebLink(batchId, modifiedFiles) {
    if (!modifiedFiles) return CVSWEB_BASE + "?batchId=" + batchId;

    // Extract first modified file for a more useful link
    const files = modifiedFiles.split(',').map(f => f.trim());
    if (files.length > 0 && files[0]) {
        // Link to diff for the first file in the commit
        return CVSWEB_BASE + "/" + files[0] + "?r1=text&r2=text";
    }
    return CVSWEB_BASE + "?batchId=" + batchId;
}

// Generate CVSweb link to view a test file's history
function makeCvswebTestFileLink(testFile) {
    if (!testFile) return null;
    // Test files are in smartclient/QA/
    // testFile format: "DataSource/advancedFilter.test" -> "isomorphic/smartclient/QA/DataSource/advancedFilter.test"
    return CVSWEB_BASE + "/isomorphic/smartclient/QA/" + testFile;
}

// Analyze a single user with full regression data
async function analyzeUser(userName, startDate, endDate) {
    console.error(`  Analyzing ${userName}...`);

    const result = {
        userName: userName,
        startDate: startDate,
        endDate: endDate,
        newTestsNeverWorked: [],
        brokenTests: [],
        wobblers: [],
        filteringApplied: TestStatusReport._getFilteringDescription()
    };

    try {
        // Fetch broken tests
        console.error(`    Fetching broken tests for ${userName}...`);
        const brokenData = await fetchBrokenByUser(dbConnection, userName, startDate, endDate);

        // Categorize results
        for (const row of brokenData) {
            // Add CVSweb link
            if (row.brokeInBatch) {
                const batch = await fetchBatchInfo(dbConnection, row.brokeInBatch);
                row.cvswebLink = batch ? makeCvswebLink(row.brokeInBatch, batch.modifiedFiles) : null;
            }

            if (row.category === 'BROKE_AND_STAYED_BROKEN') {
                result.brokenTests.push(row);
            } else if (row.category === 'BECAME_WOBBLER') {
                result.wobblers.push(row);
            }
            // RECOVERED_OR_PREEXISTING are excluded
        }

        // Fetch new tests that never worked
        console.error(`    Fetching new tests never worked for ${userName}...`);
        const newTests = await fetchNewTestsNeverWorked(dbConnection, userName, startDate, endDate);
        result.newTestsNeverWorked = newTests;

    } catch (e) {
        console.error(`    Error analyzing ${userName}: ${e.message}`);
        result.error = e.message;
    }

    return result;
}

// Format results as ASCII output
function formatResults(results, startDate, endDate) {
    const lines = [];
    const sep = '='.repeat(80);
    const thinSep = '-'.repeat(80);

    lines.push(sep);
    lines.push('TEST STATUS REPORT - REGRESSIONS BY DEVELOPER');
    lines.push('Date Range: ' + startDate + ' to ' + endDate);
    lines.push(sep);
    lines.push('');

    // Grand totals
    let grandTotal = { newBroken: 0, preBroken: 0, wobblers: 0 };

    // Summary table header
    lines.push('USER                  NEW_BROKEN  TESTS_BROKEN  WOBBLERS  TOTAL');
    lines.push(thinSep);

    // Show users with issues first
    const usersWithIssues = results.filter(r =>
        r.newTestsNeverWorked.length + r.brokenTests.length + r.wobblers.length > 0
    );
    const usersWithoutIssues = results.filter(r =>
        r.newTestsNeverWorked.length + r.brokenTests.length + r.wobblers.length === 0
    );

    for (const r of usersWithIssues) {
        const newB = r.newTestsNeverWorked.length;
        const preB = r.brokenTests.length;
        const wobb = r.wobblers.length;
        const total = newB + preB + wobb;

        grandTotal.newBroken += newB;
        grandTotal.preBroken += preB;
        grandTotal.wobblers += wobb;

        const userName = (r.userName + '                    ').slice(0, 20);
        lines.push(userName + '  ' +
                   String(newB).padStart(10) + '  ' +
                   String(preB).padStart(12) + '  ' +
                   String(wobb).padStart(8) + '  ' +
                   String(total).padStart(5));
    }

    if (usersWithoutIssues.length > 0) {
        lines.push(thinSep);
        lines.push(`${usersWithoutIssues.length} user(s) with no regressions: ${usersWithoutIssues.map(r => r.userName).join(', ')}`);
    }

    lines.push(thinSep);
    const grandTotalSum = grandTotal.newBroken + grandTotal.preBroken + grandTotal.wobblers;
    lines.push('TOTAL                 ' +
               String(grandTotal.newBroken).padStart(10) + '  ' +
               String(grandTotal.preBroken).padStart(12) + '  ' +
               String(grandTotal.wobblers).padStart(8) + '  ' +
               String(grandTotalSum).padStart(5));
    lines.push('');

    // Filtering explanation
    lines.push(thinSep);
    lines.push('FILTERING APPLIED (to exclude false positives):');
    lines.push(thinSep);
    TestStatusReport._getFilteringDescription().forEach(f => {
        lines.push('  - ' + f);
    });
    lines.push('');

    // Detailed results for users with issues
    for (const r of usersWithIssues) {
        lines.push('');
        lines.push(sep);
        lines.push('DEVELOPER: ' + r.userName);
        lines.push(sep);

        // Broken tests (pre-existing tests that broke)
        if (r.brokenTests.length > 0) {
            lines.push('');
            lines.push(thinSep);
            lines.push('TESTS BROKEN (' + r.brokenTests.length + ') - were passing, now failing:');
            lines.push(thinSep);
            for (const t of r.brokenTests) {
                lines.push('');
                lines.push('  ' + t.testFile + (t.testNumber > 1 ? ' #' + t.testNumber : ''));
                lines.push('    Broke on: ' + formatDate(t.brokeOnDate));
                lines.push('    Before commit: ' + t.passesBeforeCommit + ' pass / ' + t.failsBeforeCommit + ' fail');
                lines.push('    Since commit:  ' + t.passesSince + ' pass / ' + t.failsSince + ' fail');
                if (t.cvswebLink) {
                    lines.push('    CVSweb: ' + t.cvswebLink);
                }
            }
        }

        // New tests that never worked
        if (r.newTestsNeverWorked.length > 0) {
            lines.push('');
            lines.push(thinSep);
            lines.push('NEW TESTS NEVER WORKED (' + r.newTestsNeverWorked.length + ') - added by user, never passed:');
            lines.push(thinSep);
            for (const t of r.newTestsNeverWorked) {
                lines.push('');
                lines.push('  ' + t.testFile);
                lines.push('    First seen: ' + formatDate(t.firstSeen));
                lines.push('    Failures: ' + t.failsSince);
                // Link to test file in CVSweb
                const cvswebLink = makeCvswebTestFileLink(t.testFile);
                if (cvswebLink) {
                    lines.push('    CVSweb: ' + cvswebLink);
                }
            }
        }

        // Wobblers
        if (r.wobblers.length > 0) {
            lines.push('');
            lines.push(thinSep);
            lines.push('BECAME WOBBLERS (' + r.wobblers.length + ') - were stable, now intermittent:');
            lines.push(thinSep);
            for (const t of r.wobblers) {
                lines.push('');
                lines.push('  ' + t.testFile + (t.testNumber > 1 ? ' #' + t.testNumber : ''));
                lines.push('    Since: ' + formatDate(t.brokeOnDate) + ' | Fail rate: ' + t.failPct + '%');
                lines.push('    Before: ' + t.passesBeforeCommit + '/' + (t.passesBeforeCommit + t.failsBeforeCommit) + ' pass');
                if (t.cvswebLink) {
                    lines.push('    CVSweb: ' + t.cvswebLink);
                }
            }
        }
    }

    lines.push('');
    lines.push(sep);

    return lines.join('\n');
}

function formatDate(d) {
    if (!d) return 'N/A';
    if (typeof d === 'string') return d.slice(0, 10);
    return new Date(d).toISOString().slice(0, 10);
}

// Main function
async function main() {
    const options = parseArgs();

    if (options.help) {
        showHelp();
        process.exit(0);
    }

    // Connect to database
    try {
        console.error('Connecting to database...');
        dbConnection = await mysql.createConnection(DB_CONFIG);
    } catch (e) {
        console.error('Failed to connect to database: ' + e.message);
        process.exit(1);
    }

    try {
        // Get list of users
        let users = options.users;
        if (users.length === 0) {
            // Default: all users with commits in date range
            console.error('Fetching list of users with commits...');
            users = await getAllUsers(dbConnection, options.startDate, options.endDate);
            console.error('Found ' + users.length + ' users: ' + users.join(', '));
        }

        if (users.length === 0) {
            console.error('No users found with commits in date range.');
            process.exit(1);
        }

        // Analyze each user
        console.error('');
        console.error('Analyzing test regressions (this may take a minute)...');
        const results = [];
        for (const user of users) {
            const result = await analyzeUser(user, options.startDate, options.endDate);
            results.push(result);
        }

        // Sort by total issues (descending)
        results.sort((a, b) => {
            const aTotal = a.newTestsNeverWorked.length + a.brokenTests.length + a.wobblers.length;
            const bTotal = b.newTestsNeverWorked.length + b.brokenTests.length + b.wobblers.length;
            return bTotal - aTotal;
        });

        // Output
        console.error('');
        if (options.json) {
            console.log(JSON.stringify(results, null, 2));
        } else {
            console.log(formatResults(results, options.startDate, options.endDate));
        }
    } finally {
        await dbConnection.end();
    }
}

main().catch(e => {
    console.error('Error: ' + e.message);
    process.exit(1);
});
