/*
 * Decompiled with CFR 0.152.
 */
package com.isomorphic.datasource;

import com.isomorphic.base.Config;
import com.isomorphic.datasource.BasicDataSource;
import com.isomorphic.datasource.DSRequest;
import com.isomorphic.datasource.DSResponse;
import com.isomorphic.datasource.DataSource;
import com.isomorphic.log.Logger;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class FullTextSearchHandler {
    private static Logger log = new Logger(FullTextSearchHandler.class.getName());
    private static Config config = Config.getGlobal();
    private static final Set<String> TEXTUAL_OPERATORS = new HashSet<String>(Arrays.asList("contains", "iContains", "startsWith", "iStartsWith", "endsWith", "iEndsWith", "equals", "iEquals", "matchesPattern", "iMatchesPattern", "containsPattern", "iContainsPattern", "near", "iNear", "fuzzy", "iFuzzy", "phrase", "iPhrase", "prefixMatch", "iPrefixMatch"));
    private static final Set<String> FTS_V2_OPERATORS = new HashSet<String>(Arrays.asList("near", "iNear", "fuzzy", "iFuzzy", "phrase", "iPhrase", "prefixMatch", "iPrefixMatch"));
    private final BasicDataSource dataSource;
    private final DSRequest dsRequest;
    private final FTSConfig ftsConfig;
    private Map<String, ProviderQueryResult> providerResults;
    private List<FTSTrigger> triggers;
    private Set<Object> intersectedIds;

    public FullTextSearchHandler(BasicDataSource ds, DSRequest req) {
        this.dataSource = ds;
        this.dsRequest = req;
        this.ftsConfig = ds.getFullTextSearchConfig();
        this.providerResults = new LinkedHashMap<String, ProviderQueryResult>();
        this.triggers = new ArrayList<FTSTrigger>();
    }

    public boolean process() throws Exception {
        if (this.ftsConfig == null || !this.ftsConfig.isEnabled()) {
            return false;
        }
        if (!DataSource.isFetch(this.dsRequest.getOperationType())) {
            return false;
        }
        this.detectTriggers();
        if (this.triggers.isEmpty()) {
            log.debug("No FTS triggers detected for DataSource " + this.dataSource.getID());
            return false;
        }
        log.debug("FTS triggered for DataSource " + this.dataSource.getID() + " with " + this.triggers.size() + " trigger(s)");
        this.dsRequest.setOriginalCriteria(this.dsRequest.getRawCriteria());
        this.executeProviderQueries();
        this.intersectResults();
        if (this.intersectedIds != null && this.intersectedIds.isEmpty()) {
            log.debug("FTS intersection produced empty result set");
        }
        this.modifyCriteria();
        this.setupMergeData();
        return true;
    }

    private void detectTriggers() throws Exception {
        Object rawCriteria = this.dsRequest.getRawCriteria();
        if (rawCriteria == null || !(rawCriteria instanceof Map)) {
            return;
        }
        Map criteriaMap = (Map)rawCriteria;
        if (this.dataSource.isAdvancedCriteria(criteriaMap)) {
            this.detectTriggersInAdvancedCriteria(criteriaMap);
        } else {
            this.detectTriggersInSimpleCriteria(criteriaMap);
        }
    }

    private void detectTriggersInSimpleCriteria(Map criteria) {
        if (criteria == null) {
            return;
        }
        List<String> eligibleFields = this.dataSource.getFTSEligibleFields();
        if (eligibleFields == null || eligibleFields.isEmpty()) {
            return;
        }
        for (Object keyObj : criteria.keySet()) {
            Object value;
            String fieldName = (String)keyObj;
            if (!eligibleFields.contains(fieldName) || (value = criteria.get(fieldName)) == null || !(value instanceof String) || ((String)value).isEmpty()) continue;
            FTSTrigger trigger = new FTSTrigger();
            trigger.fieldName = fieldName;
            trigger.operator = "iContains";
            trigger.queryText = (String)value;
            trigger.provider = this.dataSource.getFTSProviderForField(fieldName);
            this.triggers.add(trigger);
            log.debug("FTS trigger (simple): field=" + fieldName + ", value=" + String.valueOf(value) + ", provider=" + trigger.provider);
        }
    }

    private void detectTriggersInAdvancedCriteria(Map criteria) {
        Object criteriaList;
        Object value;
        if (criteria == null) {
            return;
        }
        List<String> eligibleFields = this.dataSource.getFTSEligibleFields();
        if (eligibleFields == null || eligibleFields.isEmpty()) {
            return;
        }
        String operator = (String)criteria.get("operator");
        String fieldName = (String)criteria.get("fieldName");
        if (fieldName != null && eligibleFields.contains(fieldName) && operator != null && TEXTUAL_OPERATORS.contains(operator) && (value = criteria.get("value")) != null && value instanceof String && !((String)value).isEmpty()) {
            FTSTrigger trigger = new FTSTrigger();
            trigger.fieldName = fieldName;
            trigger.operator = operator;
            trigger.queryText = (String)value;
            trigger.provider = this.dataSource.getFTSProviderForField(fieldName);
            this.triggers.add(trigger);
            log.debug("FTS trigger (advanced): field=" + fieldName + ", operator=" + operator + ", value=" + String.valueOf(value) + ", provider=" + trigger.provider);
        }
        if ((criteriaList = criteria.get("criteria")) instanceof List) {
            for (Object subCriteria : (List)criteriaList) {
                if (!(subCriteria instanceof Map)) continue;
                this.detectTriggersInAdvancedCriteria((Map)subCriteria);
            }
        }
    }

    private void executeProviderQueries() throws Exception {
        ProviderRegistry registry = ProviderRegistry.getInstance();
        for (FTSTrigger trigger : this.triggers) {
            if (trigger.provider == null) {
                log.warn("No FTS provider configured for field " + trigger.fieldName);
                continue;
            }
            if (registry.isOracleTextProvider(trigger.provider)) {
                log.debug("Field " + trigger.fieldName + " uses Oracle Text - will be handled in SQL generation");
                trigger.isOracleText = true;
                continue;
            }
            try {
                ProviderQueryResult result = this.executeExternalProviderQuery(trigger);
                if (result == null) continue;
                this.providerResults.put(trigger.fieldName, result);
                trigger.result = result;
            }
            catch (Exception e) {
                this.handleProviderError(trigger, e);
            }
        }
    }

    private ProviderQueryResult executeExternalProviderQuery(FTSTrigger trigger) throws Exception {
        ProviderRegistry registry = ProviderRegistry.getInstance();
        String providerDsId = registry.getProviderDataSourceID(trigger.provider);
        if (providerDsId == null || providerDsId.isEmpty()) {
            throw new Exception("No DataSource ID configured for FTS provider: " + trigger.provider);
        }
        log.debug("Executing FTS query: provider=" + trigger.provider + ", dsId=" + providerDsId + ", field=" + trigger.fieldName + ", query=" + trigger.queryText);
        LinkedHashMap<String, Object> providerCriteria = new LinkedHashMap<String, Object>();
        providerCriteria.put("targetDataSourceId", this.dataSource.getID());
        providerCriteria.put("targetPrimaryKeyField", this.dataSource.getPrimaryKey());
        providerCriteria.put("targetFieldName", trigger.fieldName);
        providerCriteria.put("queryText", trigger.queryText);
        providerCriteria.put("operator", trigger.operator);
        providerCriteria.put("wantSnippets", Boolean.TRUE);
        providerCriteria.put("wantHighlights", Boolean.TRUE);
        long startRow = this.dsRequest.getStartRow();
        long endRow = this.dsRequest.getEndRow();
        if (startRow >= 0L) {
            providerCriteria.put("startRow", startRow);
        }
        if (endRow >= 0L) {
            providerCriteria.put("endRow", endRow);
        }
        DSRequest providerReq = new DSRequest(providerDsId, "fetch");
        providerReq.setCriteria(providerCriteria);
        providerReq.setRPCManager(this.dsRequest.getRPCManager());
        DSResponse providerResp = providerReq.execute();
        if (providerResp.statusIsError()) {
            throw new Exception("FTS provider query failed: " + providerResp.getStatus());
        }
        ProviderQueryResult result = new ProviderQueryResult();
        result.fieldName = trigger.fieldName;
        result.provider = trigger.provider;
        result.ids = new LinkedHashSet<Object>();
        result.metadata = new LinkedHashMap<Object, Map<String, Object>>();
        List records = providerResp.getDataList();
        if (records != null) {
            for (Object recordObj : records) {
                Map record;
                Object id;
                if (!(recordObj instanceof Map) || (id = (record = (Map)recordObj).get("id")) == null) continue;
                result.ids.add(id);
                LinkedHashMap meta = new LinkedHashMap();
                if (record.containsKey("score")) {
                    meta.put("score", record.get("score"));
                }
                if (record.containsKey("snippet")) {
                    meta.put("snippet", record.get("snippet"));
                }
                if (record.containsKey("snippets")) {
                    meta.put("snippets", record.get("snippets"));
                }
                if (record.containsKey("highlights")) {
                    meta.put("highlights", record.get("highlights"));
                }
                if (record.containsKey("providerMeta")) {
                    meta.put("providerMeta", record.get("providerMeta"));
                }
                if (meta.isEmpty()) continue;
                result.metadata.put(id, meta);
            }
        }
        log.debug("FTS provider returned " + result.ids.size() + " IDs for field " + trigger.fieldName);
        return result;
    }

    private void handleProviderError(FTSTrigger trigger, Exception e) throws Exception {
        boolean shouldWarn;
        ProviderRegistry registry = ProviderRegistry.getInstance();
        ProviderRegistry.ProviderConfig cfg = registry.getProviderConfig(trigger.provider);
        boolean shouldFail = cfg != null ? cfg.shouldFailOnError() : true;
        boolean bl = shouldWarn = cfg != null ? cfg.shouldWarnOnError() : false;
        if (shouldWarn) {
            log.warn("FTS provider query failed for field " + trigger.fieldName + ", provider " + trigger.provider + ": " + e.getMessage());
        }
        if (shouldFail) {
            throw new Exception("FTS provider error for field " + trigger.fieldName + ": " + e.getMessage(), e);
        }
        log.debug("Ignoring FTS provider error for field " + trigger.fieldName + ": " + e.getMessage());
    }

    private void intersectResults() {
        ArrayList<Set<Object>> idSets = new ArrayList<Set<Object>>();
        for (FTSTrigger trigger : this.triggers) {
            if (trigger.isOracleText || trigger.result == null) continue;
            idSets.add(trigger.result.ids);
        }
        if (idSets.isEmpty()) {
            return;
        }
        this.intersectedIds = new LinkedHashSet<Object>((Collection)idSets.get(0));
        for (int i = 1; i < idSets.size(); ++i) {
            this.intersectedIds.retainAll((Collection)idSets.get(i));
        }
        log.debug("FTS intersection: " + idSets.size() + " sets -> " + this.intersectedIds.size() + " IDs");
    }

    private void modifyCriteria() throws Exception {
        if (this.intersectedIds == null) {
            return;
        }
        String pkField = this.dataSource.getPrimaryKey();
        if (pkField == null) {
            log.warn("Cannot modify criteria: no primary key field for DataSource " + this.dataSource.getID());
            return;
        }
        Object[] idArray = this.intersectedIds.toArray();
        if (idArray.length == 0) {
            this.dsRequest.addToCriteria(pkField, "equals", (Object)"__FTS_NO_MATCH__");
            log.debug("FTS: added impossible criteria (empty intersection)");
        } else {
            this.dsRequest.addToCriteria(pkField, "inSet", idArray);
            log.debug("FTS: added inSet criteria with " + idArray.length + " IDs");
        }
        this.removeFTSCriteriaFromRequest();
    }

    private void removeFTSCriteriaFromRequest() {
        Object rawCriteria = this.dsRequest.getRawCriteria();
        if (rawCriteria == null || !(rawCriteria instanceof Map)) {
            return;
        }
        HashSet<String> fieldsToRemove = new HashSet<String>();
        for (FTSTrigger trigger : this.triggers) {
            if (trigger.isOracleText) continue;
            fieldsToRemove.add(trigger.fieldName);
        }
        if (fieldsToRemove.isEmpty()) {
            return;
        }
        Map criteriaMap = (Map)rawCriteria;
        if (this.dataSource.isAdvancedCriteria(criteriaMap)) {
            this.removeFTSFieldsFromAdvancedCriteria(criteriaMap, fieldsToRemove);
        } else {
            for (String fieldName : fieldsToRemove) {
                criteriaMap.remove(fieldName);
            }
        }
    }

    private void removeFTSFieldsFromAdvancedCriteria(Map criteria, Set<String> fieldsToRemove) {
        if (criteria == null) {
            return;
        }
        Object criteriaList = criteria.get("criteria");
        if (criteriaList instanceof List) {
            List list = (List)criteriaList;
            Iterator iter = list.iterator();
            while (iter.hasNext()) {
                Object subCriteria = iter.next();
                if (!(subCriteria instanceof Map)) continue;
                Map subMap = (Map)subCriteria;
                String fieldName = (String)subMap.get("fieldName");
                if (fieldName != null && fieldsToRemove.contains(fieldName)) {
                    iter.remove();
                    continue;
                }
                this.removeFTSFieldsFromAdvancedCriteria(subMap, fieldsToRemove);
            }
        }
    }

    private void setupMergeData() {
        if (this.providerResults.isEmpty()) {
            return;
        }
        LinkedHashMap aggregatedMeta = new LinkedHashMap();
        for (ProviderQueryResult providerQueryResult : this.providerResults.values()) {
            for (Map.Entry<Object, Map<String, Object>> entry : providerQueryResult.metadata.entrySet()) {
                Object id = entry.getKey();
                Map<String, Object> meta = entry.getValue();
                if (this.intersectedIds != null && !this.intersectedIds.contains(id)) continue;
                LinkedHashMap<String, Object> existing = (LinkedHashMap<String, Object>)aggregatedMeta.get(id);
                if (existing == null) {
                    existing = new LinkedHashMap<String, Object>();
                    aggregatedMeta.put(id, existing);
                }
                for (Map.Entry<String, Object> metaEntry : meta.entrySet()) {
                    String fieldType = metaEntry.getKey();
                    String actualFieldName = this.ftsConfig.getActualResultFieldName(fieldType);
                    if (existing.containsKey(actualFieldName)) continue;
                    existing.put(actualFieldName, metaEntry.getValue());
                }
            }
        }
        for (Map.Entry entry : aggregatedMeta.entrySet()) {
            this.dsRequest.addMergeData(entry.getKey(), (Map)entry.getValue());
        }
        log.debug("FTS: set up merge data for " + aggregatedMeta.size() + " records");
    }

    public List<FTSTrigger> getTriggers() {
        return this.triggers;
    }

    public boolean hasOracleTextTriggers() {
        for (FTSTrigger trigger : this.triggers) {
            if (!trigger.isOracleText) continue;
            return true;
        }
        return false;
    }

    public List<FTSTrigger> getOracleTextTriggers() {
        ArrayList<FTSTrigger> result = new ArrayList<FTSTrigger>();
        for (FTSTrigger trigger : this.triggers) {
            if (!trigger.isOracleText) continue;
            result.add(trigger);
        }
        return result;
    }

    public static class FTSConfig {
        public static final String DEFAULT_RESULT_FIELD_PREFIX = "fts_";
        public static final String DEFAULT_PROVIDER_PROPERTY = "fullText.defaultProvider";
        public static final String PROVIDERS_PROPERTY = "fullText.providers";
        public static final String ERROR_POLICY_FAIL = "fail";
        public static final String ERROR_POLICY_IGNORE = "ignore";
        public static final String ERROR_POLICY_WARN_AND_IGNORE = "warnAndIgnore";
        private boolean enabled = false;
        private String provider;
        private String resultFieldPrefix = "fts_";
        private Map<String, String> resultFieldNames;
        private String errorPolicy = "fail";
        private Map<String, Object> rawConfig;

        public static FTSConfig fromDataSourceConfig(Map<String, Object> dsConfig) {
            if (dsConfig == null) {
                return null;
            }
            Object ftsConfigObj = dsConfig.get("fullTextSearch");
            if (ftsConfigObj == null) {
                return null;
            }
            FTSConfig ftsConfig = new FTSConfig();
            ftsConfig.enabled = true;
            if (ftsConfigObj instanceof Boolean) {
                ftsConfig.enabled = (Boolean)ftsConfigObj;
            } else if (ftsConfigObj instanceof Map) {
                Object resultFieldsObj;
                Map ftsMap;
                ftsConfig.rawConfig = ftsMap = (Map)ftsConfigObj;
                if (ftsMap.containsKey("provider")) {
                    ftsConfig.provider = (String)ftsMap.get("provider");
                }
                if (ftsMap.containsKey("resultFieldPrefix")) {
                    ftsConfig.resultFieldPrefix = (String)ftsMap.get("resultFieldPrefix");
                }
                if (ftsMap.containsKey("errorPolicy")) {
                    ftsConfig.errorPolicy = (String)ftsMap.get("errorPolicy");
                }
                if ((resultFieldsObj = ftsMap.get("resultFields")) instanceof Map) {
                    ftsConfig.resultFieldNames = new HashMap<String, String>();
                    Map rfMap = (Map)resultFieldsObj;
                    for (Map.Entry entry : rfMap.entrySet()) {
                        if (entry.getValue() == null) continue;
                        ftsConfig.resultFieldNames.put((String)entry.getKey(), entry.getValue().toString());
                    }
                }
            }
            return ftsConfig;
        }

        public boolean isEnabled() {
            return this.enabled;
        }

        public String getProvider() {
            return this.provider;
        }

        public String getEffectiveProvider() {
            if (this.provider != null && !this.provider.isEmpty()) {
                return this.provider;
            }
            return config.getString(DEFAULT_PROVIDER_PROPERTY);
        }

        public String getResultFieldPrefix() {
            return this.resultFieldPrefix;
        }

        public String getResultFieldName(String fieldType) {
            if (this.resultFieldNames != null) {
                return this.resultFieldNames.get(fieldType);
            }
            return null;
        }

        public String getActualResultFieldName(String fieldType) {
            String customName = this.getResultFieldName(fieldType);
            String suffix = customName != null ? customName : fieldType;
            return this.resultFieldPrefix + suffix;
        }

        public String getErrorPolicy() {
            return this.errorPolicy;
        }

        public boolean shouldFailOnError() {
            return ERROR_POLICY_FAIL.equals(this.errorPolicy);
        }

        public boolean shouldIgnoreErrors() {
            return ERROR_POLICY_IGNORE.equals(this.errorPolicy) || ERROR_POLICY_WARN_AND_IGNORE.equals(this.errorPolicy);
        }

        public boolean shouldWarnOnError() {
            return ERROR_POLICY_WARN_AND_IGNORE.equals(this.errorPolicy);
        }

        public Map<String, Object> getRawConfig() {
            return this.rawConfig;
        }

        public static boolean isFieldExcluded(Map<String, Object> fieldConfig) {
            if (fieldConfig == null) {
                return false;
            }
            Object ftsValue = fieldConfig.get("fullTextSearch");
            if (ftsValue instanceof Boolean) {
                return (Boolean)ftsValue == false;
            }
            if (ftsValue instanceof String) {
                return "false".equalsIgnoreCase((String)ftsValue);
            }
            return false;
        }

        public static String getFieldProvider(Map<String, Object> fieldConfig) {
            if (fieldConfig == null) {
                return null;
            }
            Object provider = fieldConfig.get("fullTextProvider");
            return provider != null ? provider.toString() : null;
        }

        public static List<String> getResultFieldTypes() {
            return Arrays.asList("score", "snippet", "snippets", "highlights", "rank", "providerMeta");
        }

        public static String getResultFieldType(String fieldType) {
            switch (fieldType) {
                case "score": {
                    return "float";
                }
                case "snippet": {
                    return "text";
                }
                case "snippets": {
                    return "text";
                }
                case "highlights": {
                    return "text";
                }
                case "rank": {
                    return "integer";
                }
                case "providerMeta": {
                    return "text";
                }
            }
            return "text";
        }

        public String toString() {
            return "FTSConfig{enabled=" + this.enabled + ", provider='" + this.provider + "', resultFieldPrefix='" + this.resultFieldPrefix + "', errorPolicy='" + this.errorPolicy + "'}";
        }
    }

    public static class FTSTrigger {
        public String fieldName;
        public String operator;
        public String queryText;
        public String provider;
        public boolean isOracleText = false;
        public ProviderQueryResult result;

        public String toString() {
            return "FTSTrigger{field=" + this.fieldName + ", op=" + this.operator + ", query=" + this.queryText + ", provider=" + this.provider + ", oracleText=" + this.isOracleText + "}";
        }
    }

    public static class ProviderRegistry {
        private static Logger log = new Logger(ProviderRegistry.class.getName());
        private static Config config = Config.getGlobal();
        public static final String PROP_DEFAULT_PROVIDER = "fullText.defaultProvider";
        public static final String PROP_PROVIDERS = "fullText.providers";
        public static final String PROP_PROVIDER_PREFIX = "fullText.provider.";
        public static final String PROVIDER_TYPE_ORACLE_TEXT = "oracleText";
        public static final String PROVIDER_TYPE_EXTERNAL = "external";
        private static ProviderRegistry instance;
        private Map<String, ProviderConfig> providerConfigs;
        private String defaultProvider;
        private boolean initialized = false;

        public static synchronized ProviderRegistry getInstance() {
            if (instance == null) {
                instance = new ProviderRegistry();
            }
            return instance;
        }

        public static synchronized void reset() {
            instance = null;
        }

        private ProviderRegistry() {
        }

        private synchronized void ensureInitialized() {
            if (this.initialized) {
                return;
            }
            this.providerConfigs = new HashMap<String, ProviderConfig>();
            this.defaultProvider = config.getString(PROP_DEFAULT_PROVIDER);
            String providerList = config.getString(PROP_PROVIDERS);
            if (providerList != null && !providerList.isEmpty()) {
                String[] providers;
                for (String providerName : providers = providerList.split(",")) {
                    if ((providerName = providerName.trim()).isEmpty()) continue;
                    this.loadProviderConfig(providerName);
                }
            }
            if (this.defaultProvider != null && !this.defaultProvider.isEmpty() && !this.providerConfigs.containsKey(this.defaultProvider)) {
                this.loadProviderConfig(this.defaultProvider);
            }
            this.initialized = true;
            log.debug("FTS Provider Registry initialized: default=" + this.defaultProvider + ", providers=" + String.valueOf(this.providerConfigs.keySet()));
        }

        private void loadProviderConfig(String providerName) {
            String prefix = PROP_PROVIDER_PREFIX + providerName + ".";
            ProviderConfig cfg = new ProviderConfig();
            cfg.name = providerName;
            cfg.type = config.getString(prefix + "type", PROVIDER_TYPE_EXTERNAL);
            cfg.dataSourceID = config.getString(prefix + "dataSourceID");
            cfg.errorPolicy = config.getString(prefix + "errorPolicy", "fail");
            cfg.extraConfig = new HashMap<String, String>();
            if (PROVIDER_TYPE_EXTERNAL.equals(cfg.type) && (cfg.dataSourceID == null || cfg.dataSourceID.isEmpty())) {
                log.warn("FTS provider '" + providerName + "' is external type but has no dataSourceID configured");
            }
            this.providerConfigs.put(providerName, cfg);
            log.debug("Loaded FTS provider config: " + String.valueOf(cfg));
        }

        public String getDefaultProvider() {
            this.ensureInitialized();
            return this.defaultProvider;
        }

        public ProviderConfig getProviderConfig(String providerName) {
            this.ensureInitialized();
            if (providerName == null || providerName.isEmpty()) {
                providerName = this.defaultProvider;
            }
            return this.providerConfigs.get(providerName);
        }

        public boolean isOracleTextProvider(String providerName) {
            ProviderConfig cfg = this.getProviderConfig(providerName);
            return cfg != null && PROVIDER_TYPE_ORACLE_TEXT.equals(cfg.type);
        }

        public boolean isExternalProvider(String providerName) {
            ProviderConfig cfg = this.getProviderConfig(providerName);
            return cfg != null && PROVIDER_TYPE_EXTERNAL.equals(cfg.type);
        }

        public String getProviderDataSourceID(String providerName) {
            ProviderConfig cfg = this.getProviderConfig(providerName);
            return cfg != null ? cfg.dataSourceID : null;
        }

        public String getProviderErrorPolicy(String providerName) {
            ProviderConfig cfg = this.getProviderConfig(providerName);
            return cfg != null ? cfg.errorPolicy : "fail";
        }

        public Set<String> getProviderNames() {
            this.ensureInitialized();
            return Collections.unmodifiableSet(this.providerConfigs.keySet());
        }

        public static class ProviderConfig {
            public String name;
            public String type;
            public String dataSourceID;
            public String errorPolicy;
            public Map<String, String> extraConfig;

            public boolean isOracleText() {
                return ProviderRegistry.PROVIDER_TYPE_ORACLE_TEXT.equals(this.type);
            }

            public boolean isExternal() {
                return ProviderRegistry.PROVIDER_TYPE_EXTERNAL.equals(this.type);
            }

            public boolean shouldFailOnError() {
                return "fail".equals(this.errorPolicy);
            }

            public boolean shouldIgnoreErrors() {
                return "ignore".equals(this.errorPolicy) || "warnAndIgnore".equals(this.errorPolicy);
            }

            public boolean shouldWarnOnError() {
                return "warnAndIgnore".equals(this.errorPolicy);
            }

            public String toString() {
                return "ProviderConfig{name='" + this.name + "', type='" + this.type + "', dataSourceID='" + this.dataSourceID + "', errorPolicy='" + this.errorPolicy + "'}";
            }
        }
    }

    public static class ProviderQueryResult {
        public String fieldName;
        public String provider;
        public Set<Object> ids;
        public Map<Object, Map<String, Object>> metadata;

        public String toString() {
            return "ProviderQueryResult{field=" + this.fieldName + ", provider=" + this.provider + ", ids=" + (this.ids != null ? this.ids.size() : 0) + "}";
        }
    }
}

