# JUnit + Selenium WebDriver

[← Back to API Index](../reference.md)

---

## KB Topic: JUnit + Selenium WebDriver

### Description
Let's take a look at some JUnit code designed to test a standalone version of the [SmartClient Showcase: Trees >> Editing](http://localhost:8080/isomorphic/system/reference/SmartClient_Explorer.html#treesEditing) example. The overall test class, TreeTest, contains a test, testTree1, targeted at the TreeGrid in the example, and a test, testTree2, targeted at the SearchForm/ListGrid. As shown in the [automated testing overview](automatedTesting.md#kb-topic-automated-testing), we create a [SmartClientWebDriver](server/javadoc/com/isomorphic/webdriver/SmartClientWebDriver.html) instance which we'll use to execute commands such as [waitForElementClickable()](server/javadoc/com/isomorphic/webdriver/SmartClientWebDriver.html#waitForElementClickable-org.openqa.selenium.By-). Usage of `SmartClientWebDriver` APIs parallels Selenese commands, so you may want to review our [guide to writing Selenium scripts](usingSelenium.md#kb-topic-using-selenium-scripts-selenese) before going any further.

Just as `SmartClientWebDriver` Java calls cannot be directly generated by Selenium IDE and require [SeleneseRunner](server/javadoc/com/isomorphic/webdriver/SeleneseRunner.html), neither can we directly generate `SmartClientWebDriver`\-based JUnit4 classes. The test class TreeTest was initially generated by exporting the Selenese for testTree1 in JUnit 4 format (Java / JUnit 4 / WebDriver) to get the method declarations, but the method implementations were then filled in with Java generated by `SeleneseRunner`. Below we look at the two test cases testTree1 and testTree2.

To simplify the code for presentation, we've wrapped the driver APIs `click()` and `waitForElementClickable()` in local methods that automatically generate the [ByScLocator](server/javadoc/com/isomorphic/webdriver/ByScLocator.html). If you'd like to experiment with making changes to the sample JUnit code, one improvement that might simplify things further would be to add a `myClick()` function that handles both the `waitForElementClickable()` and the `click()` on the supplied locator. Often, just assigning each unique locator or locator prefix to a local Java variable so it can be reused for multiple calls will make the code simpler to follow and maintain.

```
 import org.openqa.selenium.*;
 import org.openqa.selenium.remote.*;
 import org.openqa.selenium.firefox.*;

 import org.junit.*;
 import static org.junit.Assert.*;
 import java.util.regex.Pattern;

 public class TreeTest {
     private SmartClientWebDriver driver;

     private SmartClientWebDriver click(String scLocator) {
         return driver.click(ByScLocator.scLocator(scLocator));
     }

     private boolean waitForElementClickable(String scLocator) {
         return driver.waitForElementClickable(ByScLocator.scLocator(scLocator));
     }
 
     @Before
     public void setUp() throws Exception {
         driver = new SmartClientFirefoxDriver();
         driver.setBaseUrl("http://localhost:8080/showcase/");
     }

     @Test
     public void testTree1() throws Exception {
         driver.get("#treesEditing");

         waitForElementClickable("scLocator=//TreeGrid[ID=\"employeeTree\"]/body/row[EmployeeId=4||Name=Charles%20Madigen||0]/col[fieldName=Name||0]/open");
         click("scLocator=//TreeGrid[ID=\"employeeTree\"]/body/row[EmployeeId=4||Name=Charles%20Madigen||0]/col[fieldName=Name||0]/open");

         waitForElementClickable("scLocator=//TreeGrid[ID=\"employeeTree\"]/body/row[EmployeeId=189||Name=Gene%20Porter||8]/col[fieldName=Name||0]/open");
         click("scLocator=//TreeGrid[ID=\"employeeTree\"]/body/row[EmployeeId=189||Name=Gene%20Porter||8]/col[fieldName=Name||0]/open");

         waitForElementClickable("scLocator=//TreeGrid[ID=\"employeeTree\"]/body/row[EmployeeId=264||Name=Cheryl%20Pearson||Salary=5650||10]/col[fieldName=Salary||2]");
         assertEquals("5650", driver.getText(ByScLocator.
             scLocator("scLocator=//TreeGrid[ID=\"employeeTree\"]/body/row[EmployeeId=264||Name=Cheryl%20Pearson||Salary=5650||10]/col[fieldName=Salary||2]")));
     }
 
```
In test testTree1, the idea is to:

*   Open the node for the top level employee, Charles Madigen,
*   Open the node for his report, Gene Porter, and
*   Verify that the Salary of Cheryl Pearson, who reports to Gene, is 5650

For this test, everything (less simplification) but the JUnit `Assert.assertEquals()` should be generated for you by `SeleneseRunner`. A verification in Selenese will likely be generated as a `verifyValue()` call to `SmartClientWebDriver`, but as that simply returns a boolean, we instead want to invoke the JUnit API directly on the result of `getText()`.

Note that though the locator for Cheryl includes the salary, it will match based on the first field, EmployeeId, which is the primary key, so the test will correctly compare the contents of Cheryl's salary against the value 5650 and fail if it doesn't match. If for some reason your test requires matching a specific field rather than the default fields and ordering generated automatically, you can hand edit the locator.

```
 
     public void testTree2() throws Exception {
         driver.get("#treesEditing");

         // Steps 1-3: Load the ListGrid with Joan's Reports
         waitForElementClickable("scLocator=//SearchForm[ID="employeeSearchForm"]/item[index=0||Class=PickTreeItem]/button/");
         click("scLocator=//SearchForm[ID="employeeSearchForm"]/item[index=0||Class=PickTreeItem]/button/");

         waitForElementClickable("scLocator=//autoID[Class=SelectionTreeMenu||index=8||length=14||classIndex=0||classLength=2||roleIndex=0||roleLength=2||scRole=menu]/body/row[Name=Charles%20Madigen]/col[fieldName=title||0]");
         mouseMove(ByScLocator.
             scLocator("scLocator=//autoID[Class=SelectionTreeMenu||index=8||length=14||classIndex=0||classLength=2||roleIndex=0||roleLength=2||scRole=menu]/body/row[Name=Charles%20Madigen]/col[fieldName=title||0]"));

         waitForElementClickable("scLocator=//SelectionTreeMenu[ID=\"isc_SelectionTreeMenu_0_childrenSubMenu_0\"]/body/row[EmployeeId=183]/col[fieldName=title||1]");
         click("scLocator=//SelectionTreeMenu[ID=\"isc_SelectionTreeMenu_0_childrenSubMenu_0\"]/body/row[EmployeeId=183]/col[fieldName=title||1]");
 
         // Step 4: Sort by salary, descending, and wait for ListGrid to be redrawn with final result
         waitForElementClickable("scLocator=//ListGrid[ID=\"employeeGrid\"]/header/headerButton[fieldName=Salary]/");
         click("scLocator=//ListGrid[ID=\"employeeGrid\"]/header/headerButton[fieldName=Salary]/");
         waitForElementClickable("scLocator=//ListGrid[ID=\"employeeGrid\"]/header/headerButton[fieldName=Salary]/");
         click("scLocator=//ListGrid[ID=\"employeeGrid\"]/header/headerButton[fieldName=Salary]/");

         driver.waitForGridDone(ByScLocator.scLocator("scLocator=//ListGrid[ID='employeeGrid']"));

         // Step 5: Verify the top salary
         waitForElementClickable("scLocator=//ListGrid[ID=\"employeeGrid\"]/body/row[0]/col[fieldName=Salary||2]");
         assertEquals("9400", selenium.getText(ByScLocator.scLocator("scLocator=//ListGrid[ID=\"employeeGrid\"]/body/row[0]/col[fieldName=Salary||2]")));
     }
 
```
In test testTree2, the idea is to:

1\. Click on the SearchForm button, revealing a Charles Madigen popup,  
2\. Issue a MouseMove on the Charles Madigen popup, revealing a list of his reports,  
3\. Click on his report Joan Little, filling the ListGrid with her reports,  
4\. Click on the salary column header twice, sorting by descending salary, and  
5\. Verify the salary in the top row (top salary) is 9400  

This test required more hand modification than the previous one. In particular three modifications were made:

*   A mouseMove command was manually added to the Selenium IDE script,
*   A call to `waitForGridDone()` was added to assure the sorting was done before we ran verifyText, and
*   We manually removed all but the row qualifier from the automatically generated scLocator for step #5.

The first modification is required because our user extensions don't record mouseMove events, and the second is needed to ensure the sorts are complete before verifyText runs--for details see the User Guide (described in [automatedTesting](automatedTesting.md#kb-topic-automated-testing)). The final modification is just a reflection of what our intent is in step #5; we want to operate on the top row, regardless of its contents, so we don't want our locator matching based on the EmployeeId or Name fields of the records. (Matching by EmployeeId in the locator as automatically generated would make the test verify that Kelly Fetterman's salary is 9400 rather than that 9400 is the highest salary.)

```
     @After
     public void tearDown() throws Exception {
         driver.quit();
     }
 }
 
```

---
