/*

  SmartClient Ajax RIA system
  Version v13.1p_2026-02-03/LGPL Deployment (2026-02-03)

  Copyright 2000 and beyond Isomorphic Software, Inc. All rights reserved.
  "SmartClient" is a trademark of Isomorphic Software, Inc.

  LICENSE NOTICE
     INSTALLATION OR USE OF THIS SOFTWARE INDICATES YOUR ACCEPTANCE OF
     ISOMORPHIC SOFTWARE LICENSE TERMS. If you have received this file
     without an accompanying Isomorphic Software license file, please
     contact licensing@isomorphic.com for details. Unauthorized copying and
     use of this software is a violation of international copyright law.

  DEVELOPMENT ONLY - DO NOT DEPLOY
     This software is provided for evaluation, training, and development
     purposes only. It may include supplementary components that are not
     licensed for deployment. The separate DEPLOY package for this release
     contains SmartClient components that are licensed for deployment.

  PROPRIETARY & PROTECTED MATERIAL
     This software contains proprietary materials that are protected by
     contract and intellectual property law. You are expressly prohibited
     from attempting to reverse engineer this software or modify this
     software for human readability.

  CONTACT ISOMORPHIC
     For more information regarding license rights and restrictions, or to
     report possible license violations, please contact Isomorphic Software
     by email (licensing@isomorphic.com) or web (www.isomorphic.com).

*/
//> @class AdaptiveMenu
// A menu that can either show its menu options inline, or show them via a drop-down,
// depending on available space in the surrounding +link{Layout} or +link{ToolStrip}.
// <p>
// See +link{canvas.canAdaptWidth,canAdaptWidth} for background on adaptive layout.
//
// @inheritsFrom Layout
// @treeLocation Client Reference/Layout
// @visibility external
//<
isc.ClassFactory.defineClass("AdaptiveMenu", "Layout");

isc.AdaptiveMenu.addProperties({

    overflow: "hidden",

    //> @attr adaptiveMenu.partialInlining (boolean : true : IRW)
    // If there is not enough space to show the full set of items as buttons inline, how
    // should the Adaptive menu behave?<br>
    // If <code>showPartialInlining</code> is true, the menu will render as many items as
    // inline buttons as can be shown in the available space, plus the menu button to 
    // access the remaining items.<br>
    // If false, it will show just the menu button.
    // <P>
    // If there is enough space to show the full set of items inline they will be shown
    // inline regardless of this property.
    //
    // @visibility external
    //<
    partialInlining: true,

    //> @attr adaptiveMenu.inlinePlacement (Alignment | VerticalAlignment : null : IR)
    // Placement of inlined items relative to the main +link{menuButton}.  Default is to place items
    // above the menu if the parent is a Layout with +link{layout.orientation,vertical orientation},
    // otherwise to the left of the <code>menuButton</code> (or right if the +link{Page.isRTL,page is
    // RTL (right-to-left)}.
    // <p>
    // A setting of "center" is invalid and will cause a warning and be ignored
    //
    // @visibility external
    //<

    //> @attr menuItem.showIconOnlyInline (boolean : null : IR)
    // When used in an +link{AdaptiveMenu}, should this MenuItem show only it's
    // +link{menuItem.icon,icon} when displayed inline?
    //
    // @visibility external
    //<

    //> @attr adaptiveMenu.showIconOnlyInline (boolean : false : IR)
    // Default setting for +link{menuItem.showIconOnlyInline}.  Individual items can set
    // <code>showIconOnlyInline</code> to override this setting.
    //
    // @visibility external
    //<
    showIconOnlyInline: false,


    // === AdaptiveMenuItem

    //> @object AdaptiveMenuItem
    // Object extending +link{object:MenuItem} and specifying an item in an 
    // +link{AdaptiveMenu}.  As with <code>MenuItems</code> each <code>AdaptiveMenuItem</code> 
    // can have a +link{menuItem.title,title}, +link{menuItem.icon,icon}, 
    // +link{menuItem.keys,shortcut keys}, optional +link{menuItem.submenu} and various other 
    // settings, including arbitrary widgets via +link{menuItem.embeddedComponent}.
    // <p>
    // Additionally, an <code>AdaptiveMenuItem</code> may be displayed either as an item in 
    // a regular +link{Menu} or as a +link{adaptiveMenu.inlineMenuItem, button in a toolstrip} 
    // and will adapt it's appearance according to layout and available space.
    // <p>
    // As a result of that behavior, the <i>menu</i> parameter passed to methods like 
    // +link{adaptiveMenuItem.click} may be null if this item is not currently displayed in  
    // a menu.  In such cases, developers may call 
    // +link{adaptiveMenuItem.getAdaptiveMenu, item.getAdaptiveMenu()}.
    //
    // @inheritsFrom MenuItem
    // @treeLocation Client Reference/Layout/AdaptiveMenu
    // @visibility external
    //<

    //> @method adaptiveMenuItem.enableIf()
    // Contains the condition that will enable or disable the current menuItem. The handler 
    // must be specified as a function or string of script.  Return false to disable the 
    // menuItem or true to enable it.
    // <p>
    // If you don't need to set this state dynamically, use +link{menuItem.enabled} instead.
    // <p>
    // Alternatively, you can use +link{Criteria} to declare when a MenuItem is
    // enabled via +link{menuItem.enableWhen}.
    // <p>
    // May be defined as a +link{group:stringMethods,stringMethod}.
    // <p>
    // Note that the <code>menu</code> parameter may be null if this 
    // <code>AdaptiveMenuItem</code> is not currently showing in a +link{class:Menu}.
    //
    // @param target (Canvas) +link{Menu.target,target} attribute for the top level menu.
    // @param menu   (Menu)   The +link{Menu} instance of which this +link{MenuItem} is a
    //                        member, or null if this item is not currently displayed in a Menu
    // @param item (AdaptiveMenuItem) contains the reference to the current item
    // @return (boolean) Return true to show a checkmark by this menu item
    //
    // @group dynamicMenuItem
    // @visibility external
    // @example menuDynamicItems
    //<

    //> @method adaptiveMenuItem.checkIf()
    // Contains the condition that will check or uncheck the current menuItem. The handler 
    // must be specified as a function or string of script.  Return false to uncheck the 
    // menuItem or true to check it
    // <p>
    // If you don't need to set this state dynamically, use +link{menuItem.checked} instead.
    // <p>
    // May be defined as a +link{group:stringMethods,stringMethod}.
    // <p>
    // Note that the <code>menu</code> parameter may be null if this 
    // <code>AdaptiveMenuItem</code> is not currently showing in a +link{class:Menu}.
    // 
    // @param target (Canvas) +link{Menu.target,target} attribute for the top level menu.
    // @param menu   (Menu)   The +link{Menu} instance of which this +link{MenuItem} is a
    //                        member, or null if this item is not currently displayed in a Menu
    // @param item (AdaptiveMenuItem) contains the reference to the current item
    // @return (boolean) Return true to show a checkmark by this menu item
    //
    // @group dynamicMenuItem
    // @visibility external
    // @example menuDynamicItems
    //<


    //> @method adaptiveMenuItem.dynamicTitle()
    // Contains the condition that will change the current items' title when met. The handler 
    // must be specified as a function or string of script.
    // <p>
    // If you don't need to set this state dynamically, use +link{menuItem.title} instead.
    // <p>
    // May be defined as a +link{group:stringMethods,stringMethod}.
    // <p>
    // Note that the <code>menu</code> parameter may be null if this 
    // <code>AdaptiveMenuItem</code> is not currently showing in a +link{class:Menu}.
    // 
    // @param target (Canvas) +link{Menu.target,target} attribute for the top level menu.
    // @param menu   (Menu)   The +link{Menu} instance of which this +link{MenuItem} is a
    //                        member, or null if this item is not currently displayed in a Menu
    // @param item (AdaptiveMenuItem) contains the reference to the current item
    // @return (String) the title of this menuItem
    //
    // @group dynamicMenuItem
    // @visibility external
    // @example menuDynamicItems
    //<


    //> @method adaptiveMenuItem.dynamicIcon()
    // Contains the condition that will change the current items' icon when met. The handler 
    // must be specified as a function or string of script.
    // <p>
    // If you don't need to set this state dynamically, use +link{menuItem.icon} instead.
    // <p>
    // May be defined as a +link{group:stringMethods,stringMethod}.
    // <p>
    // Note that the <code>menu</code> parameter may be null if this 
    // <code>AdaptiveMenuItem</code> is not currently showing in a +link{class:Menu}.
    //
    // @param target (Canvas) +link{Menu.target,target} attribute for the top level menu.
    // @param menu   (Menu)   The +link{Menu} instance of which this +link{MenuItem} is a
    //                        member, or null if this item is not currently displayed in a Menu
    // @param item (AdaptiveMenuItem) contains the reference to the current item
    // @return (SCImgURL) the url of this menuItems icon
    //
    // @group dynamicMenuItem
    // @visibility external
    // @example menuDynamicItems
    //<


    //> @method adaptiveMenuItem.click()
    // Executed when this menu item is clicked by the user. The click handler must be specified
    // as a function or string of script.  Return false to suppress the +link{Menu.itemClick()}
    // handler if specified.
    // <p>
    // Note that the <code>menu</code> parameter may be null if this 
    // <code>AdaptiveMenuItem</code> is not currently showing in a +link{class:Menu}.
    //
    // @param target (Canvas) for a menu shown as a context menu, the Canvas the menu was shown
    //                        on.  Otherwise the +link{Menu} instance of which this
    //                        +link{MenuItem} is a member.
    // @param item   (MenuItem) The +link{MenuItem} that was clicked on.
    // @param menu   (Menu)   The +link{Menu} instance of which this +link{MenuItem} is a
    //                        member, or null if this item is not currently displayed in a Menu
    // @param [colNum] (number) Index of the column the user clicked. May be null if the
    //                          user activated the menu via a keyboard event.
    // @group menuItemEvents
    // @visibility external
    //<

    //> @method adaptiveMenuItem.getAdaptiveMenu()
    // Return the +link{AdaptiveMenu} to which this item belongs    
    // @return (AdaptiveMenu) The AdaptiveMenu this item belongs to
    // @group menuItemEvents
    // @visibility external
    //<

    // ======= end AdaptiveMenuItem

    //> @attr adaptiveMenu.items (Array of MenuItem : null : IRW)
    // MenuItems to be show either inline or as a drop-down +link{Menu}.
    // <p>
    // When shown inline, items are rendered as different +link{AutoChild} according to the
    // settings on the MenuItem:
    // <ul>
    // <li> normal MenuItems render as the +link{adaptiveMenu.inlineMenuItem}, a 
    //      +link{ToolStripButton} AutoChild
    // <li> MenuItems that have submenus render as the +link{adaptiveMenu.inlineSubmenuItem}, a
    //      +link{MenuButton} AutoChild
    // <li> MenuItems with +link{menuItem.showIconOnlyInline,showIconOnlyInline} set render as 
    //      the +link{adaptiveMenu.inlineImgButton}, a +link{ToolStripButton} AutoChild
    // <li> MenuItems where +link{MenuItem.embeddedComponent} has been specified will have the
    //      embedded component displayed directly instead (no AutoChild involvement here).  If the
    //      the control should have different appearance when inlined vs embedded in the menu, 
    //      one way to achieve this is to detect whether the parent is a Menu when it is drawn.
    // <li> MenuItems with +link{menuItem.isSeparator,isSeparator} set render as
    //      the +link{adaptiveMenu.inlineSeparator}, a +link{ToolStripSeparator} AutoChild
    // </ul>
    //
    // @setter setItems (see +link{adaptiveMenu.setItems()})
    // @visibility external
    //<

    //> @method adaptiveMenu.setItems()
    //
    // @param items (Array of MenuItem | MenuItem) array of items to replace 
    //              current items
    //
    // @visibility external
    //<
    setItems : function (items) {
        return this.setData(items);
    },
    
    setData : function (items) {
        this.logDebug("Set new items", this._$adaptiveMenu);
        this.Super("setData", arguments);

        this.inlinedCount = 0;
        this.inlinedMax = 0;
        this.removedMembers = 0;

        // Now remove all members and the menuButton child
        this.setMembers([]);
        
        if (this.menuButton) this.menuButton=undefined;

        // Remove current menu entries
        this.menu = null;

        // add the getAdaptiveMenu() method to each item
        items.map(function (item) { 
            item = isc.addProperties(item, this.commonInlineMenuItemProps);
        });
        
        this.items = items;
        this.data = items;

        // If the menu is already drawn, recreate the menuButton and inline members.
        // Otherwise, this will happen when the menu is drawn.
        if (this.isDrawn()) {
            // Recreate the new inline members and menu button
            this.createChildren();

            // Refresh layout to show menu items inlined where possible
            this.refreshLayout();
        }
    },

    //> @attr adaptiveMenu.menu (AutoChild Menu : null : IR)
    // Instance of the normal (non-Adaptive) +link{Menu} class used to show items that do not fit inline.
    //
    // @visibility external
    //<
    menuDefaults: {
        _constructor: "Menu"
    },

    //> @attr adaptiveMenu.menuButton (AutoChild MenuButton : null : R)
    // +link{MenuButton} used as a drop-down control for showing any items of the menu that are not
    // displayed inline.
    //
    // @visibility external
    //<
    menuButtonConstructor: "ToolStripMenuButton",
    menuButtonDefaults: {
        overflow: "visible",
        visibility: "display"
    },

    //> @attr adaptiveMenu.menuButtonTitle (HTMLString : null : IR)
    // Title used for the +link{menuButton}.
    //
    // @visibility external
    //<

    //> @attr adaptiveMenu.menuButtonIcon (SCImgURL : null : IR)
    // Icon used for the +link{menuButton}.  Default of null means to use the default for the
    // +link{MenuButton} class.
    //
    // @visibility external
    //<

    //> @attr adaptiveMenu.inlineMenuItem (MultiAutoChild ToolStripButton : null : R)
    // +link{MultiAutoChild} used to create inline menu items.
    // <p>
    // The +link{MenuItem.icon} and +link{MenuItem.title} will be rendered via 
    // +link{Button.icon} and +link{Button.title} respectively; other +link{MenuItem} 
    // appearance-related properties do not apply.
    //
    // @visibility external
    //<
    inlineMenuItemDefaults: {
        _constructor: "ToolStripButton",
        width: 1,
        visibility: "hidden",
        wrap: false
    },

    //> @attr adaptiveMenu.inlineSubmenuItem (MultiAutoChild IconMenuButton : null : R)
    // +link{MultiAutoChild} used to create inline menu items for menu items that have a submenu.
    // <p>
    // The +link{MenuItem.icon} and +link{MenuItem.title} will be rendered via
    // +link{IconButton.icon} and +link{Button.title} respectively; other +link{MenuItem}
    // appearance-related properties do not apply.
    //
    // @visibility external
    //<
    inlineSubmenuItemDefaults: {
        _constructor: "IconMenuButton",
        width: 1,
        visibility: "hidden",
        wrap: false
    },

    //> @attr adaptiveMenu.showInlineSeparators (boolean : null : IR)
    // Whether +link{toolStripSeparator,separators} should be shown for inline menu items. 
    // True by default for horizontal +link{layout.orientation,orientation}, false for vertical.
    // <p>
    // Note, to use explicit menu separators (+link{menuItem.isSeparator}) which will also show
    // in the ToolStrip, set this property <code>false</code> to avoid showing duplicate separators in the menu.
    //
    // @visibility external
    //<

    //> @attr adaptiveMenu.separator (MultiAutoChild ToolStripSeparator : null : R)
    // +link{MultiAutoChild} used to create separators if +link{showInlineSeparators} is 
    // enabled.
    //<
    separatorDefaults: {
        isSeparator: true
    },

    //> @attr adaptiveMenu.inlineSeparator (MultiAutoChild ToolStripSeparator : null : R)
    // +link{ToolStripSeparator} to display when +link{menuItem.isSeparator,isSeparator}
    // is set for a +link{MenuItem}.
    //
    // @visibility external
    //<
    inlineSeparatorDefaults: {
        _constructor: "ToolStripSeparator",
        visibility: "hidden"
    },

    //> @attr adaptiveMenu.inlineImgButton (MultiAutoChild ImgButton : null : R)
    // +link{ToolStripButton} to display when +link{showIconOnlyInline} is set for one 
    // +link{MenuItem}
    //
    // @visibility external
    //<
    inlineImgButtonDefaults: {
        _constructor: "ToolStripButton",
        width: 1,
        title: undefined,
        visibility: "hidden",
        wrap: false
    },

    inlineEmbeddedPlaceholderDefaults: {
        _constructor: "ToolStripButton",
        height: 1,
        width: 1,
        visibility: "hidden"
    },

    // Properties to be applied to menuItems
    commonInlineMenuItemProps: {
        getAdaptiveMenu : function () {
            var adaptiveMenu = this._member ? this._member.creator : this.creator;
            if (isc.isA.Menu(adaptiveMenu)) adaptiveMenu = adaptiveMenu.creator;
            return adaptiveMenu;
        }
    },
    
    // Properties to be applied to each inline button type
    commonInlineButtonProps: {
        getAdaptiveMenu : function () {
            var adaptiveMenu = this.creator;
            if (isc.isA.Menu(adaptiveMenu)) adaptiveMenu = adaptiveMenu.creator;
            return adaptiveMenu;
        },
        processVisibilityRule : function (result, rule) {
            // Ignore rule processing that occurs before all child objects are created
            if (!this.creator._initializedChildren) {
                return;
            }

            var item = this._item;

            // Mark item with status of rule visibility. This is used to filter out
            // inline buttons that are hidden because of a rule.
            var targetVisible = !!result;
            item._visibleWhenHidden = !targetVisible;

            // If showing a button that is hidden because of the adaptive
            // placement algorithm, there is nothing more to do.
            if (targetVisible && item._adaptiveHidden) {
                // Item is hidden because of layout - don't change it
                return;
            }

            // If visibility is changing, refresh menu with adaptive layout refresh.
            // Don't actually show/hide an inline button because that can cause a
            // temporary flash of buttons until the results are reflowed.
            var currentVisible = (this.visibility != "hidden");
            if (currentVisible != targetVisible) {
                // Refresh the layout on pause so multiple updates will only recalculate once.
                // Use the class method so calls on different buttons are consolidated to
                // one refresh.
                if (!this.creator._pendingRefreshLayout) {
                    this.creator.logDebug("Schedule refresh layout because an inlined item (" + this.title + ") visibility changed", this.creator._$adaptiveMenu);
                    this.creator._pendingRefreshLayout = true;
                }
                isc.Class.fireOnPause(
                    "refreshAdaptiveMenu",
                    {target:this.creator, methodName:"refreshLayout"},
                    10
                );
                return true;
            }
        }
    }
});

// add instance methods
isc.AdaptiveMenu.addMethods({

    _$adaptiveMenu:"adaptiveMenu",

    initWidget : function () {
        this.inlinedCount = 0;
        this.removedMembers = 0;

        if (this.vertical) {
            this.canAdaptHeight = true;
            if (this.showInlineSeparators == undefined) this.showInlineSeparators = false;
        } else {
            this.canAdaptWidth = true;
            if (this.showInlineSeparators == undefined) this.showInlineSeparators = true;
            // For H-Layouts perform a horizontal animation effect when showing / hiding
            this.animateMemberEffect = {effect:"slide", startFrom:"L", endAt:"L"};
        }

        // Check inlinePlacement: A setting of "center" is invalid and will cause a warning and be ignored
        if (this.inlinePlacement == "center") {
            isc.logWarn("Center is an invalid value for inlinePlacement in adaptiveMenu, ignoring this setting.");
            this.inlinePlacement = undefined;
        }

        // add the getAdaptiveMenu() method to each item
        if (this.items) {
            var _this = this;
            this.items.map(function (item) { 
                item = isc.addProperties(item, _this.commonInlineMenuItemProps);
            });
        }

       // call the superclass function
       this.Super("initWidget",arguments);
    },
    
    createChildren : function () {
        for (var i = 0; i < this.items.getLength(); i++) {
            var item = this.items.get(i);
            var showIconOnlyInline =
                item.showIconOnlyInline == undefined ? this.showIconOnlyInline : item.showIconOnlyInline;

            var lastMember;

            if (item.isSeparator) {
                lastMember = this.createAutoChild("inlineSeparator", {
                    vertical: !this.vertical,
                    // apply skin sizes
                    height: (!this.vertical ? isc.ToolStrip.getInstanceProperty("separatorSize") : isc.ToolStrip.getInstanceProperty("separatorBreadth")),
                    width: (!this.vertical ? isc.ToolStrip.getInstanceProperty("separatorBreadth") : isc.ToolStrip.getInstanceProperty("separatorSize"))
                });
            } else if (item.embeddedComponent) {
                // Embedded component starts life in the menu but to keep
                // the members matching, a (hidden) placeholder is added
                // in its place.
                item._placeholder = this.createAutoChild("inlineEmbeddedPlaceholder", { _item: item });
                lastMember = item.embeddedComponent;
                lastMember.visibility = "hidden";
            } else if (showIconOnlyInline) {
                lastMember =  this.createAutoChild("inlineImgButton", isc.addProperties({
                    enableWhen:item.enableWhen, visibleWhen:item.visibleWhen,
                    icon: item.icon,
                    click: item.click
                }, this.commonInlineButtonProps));
            } else if (item.submenu != undefined) {

                var menu = this.createAutoChild("inlineSubmenu", {data: item.submenu}, isc.Menu);
                lastMember = this.createAutoChild("inlineSubmenuItem", isc.addProperties({
                    enableWhen:item.enableWhen, 
                    visibleWhen:item.visibleWhen,
                    icon: item.icon,
                    title: item.title,
                    click: item.click,
                    menu: menu
                }, this.commonInlineButtonProps));
            } else {
                lastMember = this.createAutoChild("inlineMenuItem", isc.addProperties({
                    enableWhen:item.enableWhen, 
                    visibleWhen:item.visibleWhen,
                    icon: item.icon,
                    title: item.title,
                    click: item.click
                }, this.commonInlineButtonProps));
            }
            this.addMember(lastMember);

            // Add references between items and members so they can be resolved easily
            lastMember._item = item;
            item._member = lastMember;
            // Button defaults to hidden; item is marked to match
            item._adaptiveHidden = true;
        }
        this.inlinedMax = this.items.getLength();

        if (!this.menuButton) this.createMenuButton();
        else if (!this.menuButton.isDrawn()) this.menuButton.draw();

        this._initializedChildren = true;
    },

    // Lazily create and populate the menu
    populateMenu : function (rebuild) {
        if (!this.menuButton.menu || rebuild) {
            var data = [];
            var j = 0;
            for (var i = 0; i < this.items.getLength(); i++) {
                var item = this.items.get(i);
                var member = item._member;
                var component = item.embeddedComponent ? item.embeddedComponent : member;

                if (rebuild && (component.visibility !== "hidden" || item._visibleWhenHidden)) {
                    continue;
                }
                data[j++] = item;
                if (this.showInlineSeparators && i < this.items.getLength()-1) data[j++] = this.createAutoChild("separator");
            }

            this.menu = this.createAutoChild("menu", {
                data: data
            });

            this.menuButton.menu = this.menu;
        }

        return this.menu;
    },

    createMenuButton : function () {
        // MenuButton autochild should be just barely wider than its icon by default
        var mbWidth = 1;
        if (this.menuButtonIcon) mbWidth = this.menuButtonIcon.width + 1;
        
        this.menuButton = this.createAutoChild("menuButton", {
                title: this.menuButtonTitle,
                icon: this.menuButtonIcon,
                width: mbWidth
            });

        // Draw 'menuButton' offscreen to get its width and height for later use.
        // This is important because the first time the menuButton is shown it
        // may not have a valid visible width/height. _showMenuButton() will
        // reference the correct dimension value as needed.
        isc.Canvas.moveOffscreen(this.menuButton);
        this.menuButton.show();
        this.menuButtonWidth = this.menuButton.getVisibleWidth();
        this.menuButtonHeight = this.menuButton.getVisibleHeight();
        this.menuButton.hide();

        // Add the button to out inlined items
        this.addMember(this.menuButton);

        

        // Observe menuButton, to avoid creating Menu until drop-down is clicked
        this.observe(this.menuButton, "click", function () {
            // Create and draw the menu, if not created before
            this.populateMenu();
        });
    },

    // Avoid creating MenuButton or any inlineItem AutoChildren
    // until first call to layoutChildren()
    layoutChildren : function (reason) {
        this.Super("layoutChildren", arguments);
        if (!this._initializedChildren) {
            this.createChildren();
        }
    },

    refreshLayout : function () {
        this.logDebug("Refresh layout", this._$adaptiveMenu);
        delete this._pendingRefreshLayout;

        var _this = this;
        var hideEmbeddedComponent = function (item) {
            if (!item || !item.embeddedComponent) return;

            item.embeddedComponent.hide();
            item._adaptiveHidden = true;
            delete item.embeddedComponent.embeddedPosition;

            // Add the placeholder at the embedded component's position
            var memberNumber = _this.getMemberNumber(item.embeddedComponent);
            if (memberNumber >= 0) {
                _this.removeMember(item.embeddedComponent);
                _this.addMember(item._placeholder, memberNumber);
            }

            if (item.embeddedComponent._snapTo) {
                item.embeddedComponent.setSnapTo(item.embeddedComponent._snapTo);
                delete item.embeddedComponent._snapTo;
            }

            // Remove the embedded component to the menu
            if (_this.menu) _this.menu.updateRecordComponents();
        };

        // To refresh, hide existing inline buttons and rebuild the menu with
        // items that should be shown. Then allow the adaptive sizing to occur
        // placing items as inline buttons while there is room.
        var members = this.members;
        for (var i = 0; i < members.length; i++) {
            var member = this.members[i];
            var item = member._item;
            var component = item && item.embeddedComponent ? item.embeddedComponent : member;

            if (component && component !== this.menuButton) {
                component.hide();
                item._adaptiveHidden = true;
            }

            // If the item we just added was an embedded component, tell the menu to update its
            // record components. This will reparent the embedded component to the menu.
            if (item && item.embeddedComponent) {
                hideEmbeddedComponent(item);
            }
        }
        // Hide all embedded components not currently inlined as well
        for (i = 0; i < this.inlinedMax; i++) {
            item = this.items.get(i);
            if (item && item.embeddedComponent) {
                hideEmbeddedComponent(item);
            }
        }

        // Reset inlinedCount to match the non-hidden inline items: 0
        this.inlinedCount = 0;

        // Rebuild the menu
        this.populateMenu(true);

        // Assign menu items to inline buttons where applicable
        if (!this.vertical) this.adaptWidthBy(0, this.getVisibleWidth());
        else this.adaptHeightBy(0, this.getVisibleHeight());
    },

    //> @method AdaptiveMenu.setPartialInlining()
    //
    // @param partialInlining (boolean) 
    // @visibility external
    //<
    setPartialInlining : function (partialInlining) {
        this.logDebug("Partial inlining changed from " + this.partialInlining + " to " + partialInlining, this._$adaptiveMenu);
        this.partialInlining = partialInlining;
        
        this.refreshLayout();
    },

    adaptWidthBy : function (pixelDifference, unadaptedWidth) {
        return this._adaptDimensionBy("Width", pixelDifference, unadaptedWidth);
    },

    adaptHeightBy : function (pixelDifference, unadaptedHeight) {
        return this._adaptDimensionBy("Height", pixelDifference, unadaptedHeight);
    },

    // get dimension of the next item to be inlined, by drawing it if needed
    // axis needs to be either "Width" or "Height" case sensitive.
    getNextInlineItemDimension : function (dimension, excludeMinimal) {
        var itemPosition = isc.NumberUtil.clamp(this.inlinedCount, 0, this.inlinedMax);
        var item = this.items.get(this.inlinedCount);
        var member = item._member;
        var component = item && item.embeddedComponent ? item.embeddedComponent : member;

        if (!component.isDrawn()) component.draw();
        if (item._visibleWhenHidden) return 0;
        var isLast = itemPosition === this.inlinedMax - 1;
        var value = component["getVisible" + dimension]() + (isLast ? -this["minimal" + dimension] : 0);
        return component["getVisible" + dimension]() + (!excludeMinimal && isLast ? -this["minimal" + dimension] : 0);
    },

    getNextInlineItem : function () {
        return this.items.get(this.inlinedCount);
    },

    // get width of the next item to be inlined, by drawing it if needed
    getNextInlinedItemWidth : function () {
        return this.getNextInlineItemDimension("Width");
    },
    
    // get height of the next item to be inlined, by drawing it if needed
    getNextInlinedItemHeight : function () {
        return this.getNextInlineItemDimension("Height");
    },

    getItemDimension : function (item, dimension) {
        var component = item.embeddedComponent ? item.embeddedComponent : item._member;
        // if (!component.isDrawn()) component.draw();
        return (item._visibleWhenHidden ? 0 : component["getVisible" + dimension]());
    },

    // remove an inlined item - show menu button if appropriate.
    //
    // Members array is always the same number of components matching the 'items' array.
    // For an embedded component, it can only be in the inlined toolstrip or the menu at any given
    // time so it is actually moved to the menu and a hidden placeholder holds its place in the
    // toolstrip.
    //
    // Removing an inlined item means hiding the toolstrip button and adding the corresponding
    // menu item.
    //
    // @return (Integer) the number of pixels removed or -1 if nothing could be done
    //
    removeInlinedItem : function (dimension) {
        var currentMenu = this.populateMenu();
        if (!currentMenu) {
            return -1;
        }

        if (this.inlinedCount === 0) {
            return -1;
        }

        // When showing explicit inline separators, push the separator
        // into the menu before the item itself. Menu shouldn't end in
        // a separator.
        if (this.showInlineSeparators && currentMenu.data.getLength() > 0) {
            currentMenu.data.addAt({isSeparator: true}, 0);
        }

        var item = this.items[this.inlinedCount-1];

        if (!item) return -1;      // should not occur

        // Update the inlined count of buttons
        this.inlinedCount--;

        // Hide the inlined button
        // When hiding a component because of the layout size, mark it with a special
        // flag which can be used by visibleWhen rule processing (processVisibilityRule)
        // to avoid showing it.
        var member = item._member;
        var component = item.embeddedComponent ? item.embeddedComponent : member;
        component.hide();
        item._adaptiveHidden = true;

        // If an explicit separator is just added to the top of the menu, it will show. Instead,
        // record there is a pending item (separator) to be either added to the menu on the next
        // item added or pulled off when removing items from menu.
        if (item.isSeparator) {
            currentMenu._pendingItem = item;
        } else {
            if (currentMenu._pendingItem) {
                currentMenu.data.addAt(currentMenu._pendingItem, 0);
                delete currentMenu._pendingItem;
            }
            currentMenu.data.addAt(item, 0);
        }

        // Special handling of item enable state being managed by rule. The inlined button has the
        // correct state and, otherwise, all rules need to be re-evaluated to update the new menu item.
        if (item.enableWhen) {
            currentMenu.setItemEnabled(0, !component.disabled);
        }

        // If the item we just added was an embedded component, tell the menu to update its
        // record components. This will reparent the embedded component to the menu.
        if (item.embeddedComponent) {
            // Add the placeholder at the embedded component's position
            var memberNumber = this.getMemberNumber(item.embeddedComponent);
            this.removeMember(item.embeddedComponent);
            this.addMember(item._placeholder, memberNumber);

            if (item.embeddedComponent._snapTo) {
                item.embeddedComponent.setSnapTo(item.embeddedComponent._snapTo);
                delete item.embeddedComponent._snapTo;
            }

            // Remove the embedded component to the menu
            currentMenu.updateRecordComponents();

            
        }

        var delta = this.getItemDimension(item, dimension);

        if (!item.isSeparator) {
            this.logDebug("Removed inlined item (" + delta + "): " + (item.embeddedComponent ? "embedded component" : item.title), this._$adaptiveMenu);
            var newLastItem = this._getLastInlinedItem();
            if (newLastItem && newLastItem.isSeparator) {
                var separatorDelta = this.removeInlinedItem(dimension);
                delta += separatorDelta;
            }
        } else {
            this.logDebug("Removed inlined item (" + delta + "): separator", this._$adaptiveMenu);
        }

        return delta;
    },

    // Find 'inlinedCount' item in the inlined buttons. Because of rules hiding
    // individual buttons, these may not be found by index.
    _getLastInlinedItem : function () {
        var item;
        for (var i = this.inlinedCount - 1; i >= 0 ; i--) {
            if (!this.items[i]._adaptiveHidden) {
                item = this.items[i];
                break;
            }
        }
        return item;
    },

    // add an inlined item - hide menu button if appropriate
    //
    // @return (Integer) delta in pixels added
    addInlinedItem : function (dimension, remainingSpace) {
        var currentMenu = this.populateMenu();

        if (!currentMenu) {
            return;
        }

        var item = this.items[this.inlinedCount],
            itemSize = this.getItemDimension(item, dimension)
        ;
        if (itemSize > remainingSpace) {
            return -1;
        }

        // If an explicit separator pending at the top of the menu, use that. Otherwise, grab the
        // top-most menu item. Note that a menu will not include a visibleWhen-hidden item at all,
        // therefore, the item checked above may not match the item to be removed from the menu.
        var itemRemovedFromMenu = null;
        if (currentMenu._pendingItem) {
            itemRemovedFromMenu = currentMenu._pendingItem;
        } else {
            itemRemovedFromMenu = currentMenu.data.getLength() > 0 ? currentMenu.data.get(0) : null;
        }

        if (itemRemovedFromMenu && itemRemovedFromMenu === item) {
            if (currentMenu._pendingItem) {
                delete currentMenu._pendingItem;
            } else {
                currentMenu.data.removeAt(0);
            }
        } else {
            itemRemovedFromMenu = item;
        }

        if (this.showInlineSeparators && currentMenu.data.getLength() > 0) {
            currentMenu.data.removeAt(0);
        }

        // If the menu item we just removed has an embedded component, let's ensure that record
        // components are updated on the menu.
        if (itemRemovedFromMenu && itemRemovedFromMenu.embeddedComponent) {
            currentMenu.removeEmbeddedComponent(itemRemovedFromMenu, itemRemovedFromMenu.embeddedComponent);
            currentMenu.updateRecordComponents();

            // We need to tell the menu to redraw here, this will update the markup in order to
            // be able to add the embedded component back to the AdaptiveMenu later on.
            if (currentMenu.isDrawn()) {
                currentMenu.redraw();
            }
        }

        var itemToAdd = itemRemovedFromMenu;

        // An embeddedComponent item cannot be in both the menu and the toolstrip.
        // Move these back into the toolstrip always placing them in the same
        // position as defined in the 'items'. This is because members of the toolstrip
        // are always in the same positions and never move around so any visibleWhen
        // rules will just hide/show the item as needed.
        if (itemToAdd && itemToAdd.embeddedComponent) {
            if (itemToAdd.embeddedComponent.snapTo) {
                itemToAdd.embeddedComponent._snapTo = itemToAdd.embeddedComponent.snapTo;
                delete itemToAdd.embeddedComponent.snapTo;
            }

            var position = this.getMemberNumber(itemToAdd._placeholder);
            if (position === -1) position = this.inlinedCount;
            this.removeMember(itemToAdd._placeholder);
            this.addMember(itemToAdd.embeddedComponent, position);
        }

        var member = itemToAdd && itemToAdd._member;
        if (member && !itemToAdd._visibleWhenHidden) member.show();
        if (itemToAdd) {
            // No longer hidden
            delete itemToAdd._adaptiveHidden;
            var size = (itemToAdd._visibleWhenHidden ? 0 : member["getVisible" + dimension]());
            this.logDebug("Added inline item: " + (itemToAdd.embeddedComponent ? "embedded component" : (itemToAdd.isSeparator ? "separator" : itemToAdd.title)) + " (" + size + ")", this._$adaptiveMenu);

            this.inlinedCount++;
        }

        // If there are no items left in the menu, lets hide the menu button.
        if (currentMenu.getTotalRows() === 0) {
            this.menuButton.hide();
            this.logDebug("Hide menu button", this._$adaptiveMenu);
        }

        return itemSize;
    },

    _adaptDimensionBy : function (dimension, pixelDifference, unadaptedValue) {
        // all non-hidden children are drawn; expected height is sum of their Heights
        var expectedDimension = 0;  // Current Height of all members, included menu if visible
        var inlinedDimension = 0;   // Current Height of all inlined elements, visible or not, menu excluded
        var getComponent = function (item) {
            return item && item.embeddedComponent ? item.embeddedComponent : item._member;
        };
        var isComponentHidden = function (item, component) {
            return ((component && component.visibility === "hidden") ||
                      (item && item.embeddedComponent && !item.embeddedComponent.isVisible()));
        };
        for (var i = 0; i < this.inlinedMax; i++) {
            var item = this.items.get(i);
            var component = getComponent(item);

            if (component !== this.menuButton) {
                if (!component.isDrawn()) {
                    component.draw();
                }
                inlinedDimension += component["getVisible" + dimension]();
            }

            if (isComponentHidden(item, component)) {
                continue;
            }

            expectedDimension += component["getVisible" + dimension]();
        }

        // calculate desired Height based on overflow/surplus and unadapted Height;
        // if desired Height differs from the expected, add/remove inlined items
        var desiredDimension = unadaptedValue + pixelDifference;

        if (!this.partialInlining) {
            if (inlinedDimension > desiredDimension) {
                // Remove all inlined
                while (this.inlinedCount > 0) {
                    this.removeInlinedItem(dimension);
                }
                this.menuButton.show();
                return this.menuButton["getVisible" + dimension]() - unadaptedValue;
            }
        }

        this.logDebug("Adapt menu by " + dimension + " by " + pixelDifference + " (" + unadaptedValue + ")", this._$adaptiveMenu);
        this.logDebug("Available space: " + desiredDimension + ", used space: " + expectedDimension, this._$adaptiveMenu);
        this._dumpVisibleItems("Current", dimension);

        if (desiredDimension < expectedDimension) {
            // remove inlined items if we have an overflow
            while (this.inlinedCount > 0 && expectedDimension > desiredDimension) {
                var delta = this.removeInlinedItem(dimension);
                if (delta < 0) {
                    // No more items found to remove
                    break;
                }
                expectedDimension -= delta;
            }
        } else if (desiredDimension > expectedDimension) {
            // add inlined items if we have surplus space

            while (this.inlinedCount < this.inlinedMax && expectedDimension < desiredDimension) {
                var delta = this.addInlinedItem(dimension, desiredDimension - expectedDimension);
                if (delta < 0) {
                    // No more items to be added or next item is larger than available space
                    break;
                }
                expectedDimension += delta;
            }

            // Don't stop with a trailing separator
            var newLastItem = this._getLastInlinedItem();
            if (newLastItem && newLastItem.isSeparator) {
                expectedDimension -= this.removeInlinedItem(dimension);
            }
        }

        // If the menu has items that were not inlined, show the menu button
        if (this.menu && this.menu.getTotalRows() > 0) {
            this._showMenuButton(dimension, expectedDimension, desiredDimension);
        }

        this._dumpVisibleItems("Done adapt " + dimension, dimension);

        return pixelDifference;
    },

    _showMenuButton : function (dimension, expectedDimension, desiredDimension) {
        // If menuButton doesn't fit in the remaining dimension, remove additional items until it will
        var menuButtonDimension = this["menuButton" + dimension];
        if (menuButtonDimension == null) {
            this.logError("MenuButton has no visible dimension. Cannot be shown.");
        }

        if (menuButtonDimension + expectedDimension > desiredDimension) {
            this.logDebug("Make space for menu button (" + menuButtonDimension + "). " + menuButtonDimension + "+" + expectedDimension + " > " + desiredDimension, this._$adaptiveMenu);
            expectedDimension += menuButtonDimension;

            // remove inlined items if we have an overflow
            while (this.inlinedCount > 0 && expectedDimension > desiredDimension) {
                var delta = this.removeInlinedItem(dimension);
                if (delta < 0) {
                    // No more items found to remove
                    break;
                }
                expectedDimension -= delta;
            }
        }

        this.menuButton.show();
        this.logDebug("Show menu button", this._$adaptiveMenu);

        // Any remaining menu items should be marked as _adaptiveHidden so rules
        // will not show them
        for (var i = this.inlinedCount; i < this.inlinedMax; i++) {
            var item = this.items.get(i);
            item._adaptiveHidden = true;
        }
    },

    _dumpVisibleItems : function (prefix, dimension) {
        if (!this.logIsDebugEnabled(this._$adaptiveMenu)) return;
        dimension = dimension || "Width";
        var items = this.items,
            totalSize = 0,
            itemsMessage = "";
        for (var i = 0; i < items.length; i++) {
            var item = items[i],
                member = item._member;
            if (!item._adaptiveHidden) {
                var size = (item._visibleWhenHidden ? 0 : member["getVisible" + dimension]());
                itemsMessage += " " + (item.embeddedComponent ? "embedded component" : (item.isSeparator ? "separator" : item.title)) + " (" + size + ")";
                totalSize += size;
            }
        }
        this.logDebug(prefix + " inlined items [" + this.inlinedCount + " of " + this.inlinedMax + ", total size=" + totalSize + "]: " + itemsMessage, this._$adaptiveMenu);
    },

    // For auto-tests
    getVisibleInlinedMemberCount : function () {
        var membersCount = this.getMembers().length,
            count = 0;
        for (var i = 0; i < membersCount; i++) {
            var member = this.getMember(i);
            if (member.visibility != "hidden" && member != this.menuButton) {
                count++;
            }
        }
        return count;
    },

    // For auto-tests
    getInlinedItemCount : function () {
        var items = this.items,
            count = 0;
        for (var i = 0; i < items.length; i++) {
            var item = items[i],
                member = item._member;
            if (!item._adaptiveHidden) {
                count++;
            }
        }
        return count;
    },

    // For auto-tests
    getVisibleItemsTotalLength : function () {
        var dimension = this.vertical ? "Height" : "Width",
            items = this.items,
            totalSize = 0;
        for (var i = 0; i < items.length; i++) {
            var item = items[i],
                member = item._member;
            if (!item._adaptiveHidden) {
                var size = (item._visibleWhenHidden ? 0 : member["getVisible" + dimension]());
                totalSize += (item._visibleWhenHidden ? 0 : member["getVisible" + dimension]());
            }
        }
        return totalSize;
    }
});
